// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.generics.util;

import de.caff.annotation.NotNull;

/**
 *  Changeable integer wrapper, to be used as counter especially as map values.
 *  Note that as this is a mutable value, it is a very bad idea to use it as a map key.
 * <p>
 *  This class is not thread safe, see {@link ThreadSafeLongCounter} for a thread safe counter!
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class SimpleLongCounter
        implements LongCounter
{
  private long value;

  /**
   *  Default constructor. Initialize value with 0.
   */
  public SimpleLongCounter() {
    value = 0;
  }

  /**
   *  Constructor. Initialize value with {@code val}.
   *  @param   val  init value
   */
  public SimpleLongCounter(int val) {
    value = val;
  }

  /**
   *  Copy constructor. Initialize value with given counter.
   *  @param  count  counter to be copied
   */
  public SimpleLongCounter(@NotNull LongCounter count) {
    value = count.getValue();
  }

  @Override
  public long add(long add)
  {
    return value += add;
  }

  @Override
  public long add(@NotNull LongCounter counter)
  {
    return value += counter.getValue();
  }

  @Override
  public long add1()
  {
    return ++value;
  }

  @Override
  public long subtract(long sub)
  {
    return value -= sub;
  }

  @Override
  public long subtract(@NotNull LongCounter counter)
  {
    return value -= counter.getValue();
  }

  @Override
  public long subtract1()
  {
    return --value;
  }

  @Override
  public long getValue() {
    return value;
  }

  @Override
  public void setValue(long value)
  {
    this.value = value;
  }

  @Override
  @NotNull
  public String toString()
  {
    return Long.toString(value);
  }
}
