// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.generics.util;

import de.caff.annotation.NotNull;

/**
 *  Changeable integer wrapper, to be used as counter especially as map values.
 *  Note that as this is a mutable value, it is a very bad idea to use it as a map key.
 * <p>
 *  This class is not thread safe, see {@link ThreadSafeCounter} for a thread safe counter!
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class SimpleCounter
        implements Counter
{
  private int value;

  /**
   *  Default constructor. Initialize value with 0.
   */
  public SimpleCounter() {
    value = 0;
  }

  /**
   *  Constructor. Initialize value with {@code val}.
   *  @param   val  init value
   */
  public SimpleCounter(int val) {
    value = val;
  }

  /**
   *  Copy constructor. Initialize value with given counter.
   *  @param  count  counter to be copied
   */
  public SimpleCounter(@NotNull Counter count) {
    value = count.getValue();
  }

  /**
   * Add something to the counter.
   *
   * @param add value to add
   * @return counter value
   */
  @Override
  public int add(int add)
  {
    return value += add;
  }

  /**
   * Add something to the counter.
   *
   * @param counter counter which value is added
   * @return counter value
   */
  @Override
  public int add(@NotNull Counter counter)
  {
    return value += counter.getValue();
  }

  /**
   * Add 1 to counter.
   *
   * @return counter value
   */
  @Override
  public int add1()
  {
    return ++value;
  }

  /**
   * Subtract something from the counter.
   *
   * @param sub value to subtract
   * @return counter value
   */
  @Override
  public int subtract(int sub)
  {
    return value -= sub;
  }

  /**
   * Subtract something from the counter.
   *
   * @param counter counter which value is subtracted
   * @return counter value
   */
  @Override
  public int subtract(@NotNull Counter counter)
  {
    return value -= counter.getValue();
  }

  /**
   * Subtract 1 from counter.
   *
   * @return counter value
   */
  @Override
  public int subtract1()
  {
    return --value;
  }

  /**
   *  Get the value.
   *  @return  accumulated value
   */
  @Override
  public int getValue() {
    return value;
  }

  /**
   * Set the value.
   *
   * @param value new value
   */
  @Override
  public void setValue(int value)
  {
    this.value = value;
  }

  @Override
  public String toString()
  {
    return Integer.toString(value);
  }
}
