// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.util;

import de.caff.annotation.NotNull;

/**
 * Reference type.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public enum ReferenceType
{
  /** Hard reference. */
  Hard(HardReferenceImpl::new),
  /**
   * Soft reference.
   * @see java.lang.ref.SoftReference
   */
  Soft(SoftReferenceImpl::new),
  /**
   * Weak reference.
   * @see java.lang.ref.WeakReference
   */
  Weak(WeakReferenceImpl::new);


  private final ReferenceCreator referenceCreator;

  /**
   * Constructor.
   * @param referenceCreator reference creator delegate
   */
  ReferenceType(ReferenceCreator referenceCreator)
  {
    this.referenceCreator = referenceCreator;
  }

  /**
   * Make a reference from an object.
   * @param obj basic object
   * @param <T> object type
   * @return a reference to the object
   */
  @NotNull
  public <T> IReference<T> makeReference(@NotNull T obj)
  {
    return referenceCreator.makeReference(obj);
  }

  /** Helper interface for reference creation. */
  private interface ReferenceCreator
  {
    /**
     * Create a reference to an object.
     * @param obj object to which a reference is created
     * @param <T> reference type
     * @return reference to object
     */
    @NotNull
    <T> IReference<T> makeReference(@NotNull T obj);
  }
}
