// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.util;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.util.Objects;

/**
 * A mutable reference to an object.
 * This is sometimes useful to return a value from lambdas, although it is usually considered a hack.
 * It is not thread-safe!
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 10, 2022
 * @param <T> type of object kept by this reference
 */
public class MutableRef<T>
        implements IReference<T>
{
  private T reference;

  /**
   * Constructor creating a {@code null} reference
   */
  public MutableRef()
  {
  }

  /**
   * Constructor.
   * @param reference referenced object
   */
  public MutableRef(@Nullable T reference)
  {
    this.reference = reference;
  }

  /**
   * Set the reference.
   * @param reference referenced object
   */
  public void set(@Nullable T reference)
  {
    this.reference = reference;
  }

  @Nullable
  @Override
  public T get()
  {
    return reference;
  }

  /**
   * Require that the reference value is set to a non-null value.
   * @return referenced value
   */
  @NotNull
  public T require()
  {
    return Objects.requireNonNull(reference);
  }
}
