// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.util;

import de.caff.annotation.NotNull;

/**
 * A long counter is basically a long integer which can be used for counting.
 * <p>
 * Depending on the usage it comes in a
 * {@link ThreadSafeLongCounter thread-safe implementation}
 * and a {@link SimpleLongCounter non thread-safe implementation}.
 * <p>
 * There are also factories for both counters:
 * {@link LongCounter#SIMPLE} and {@link LongCounter#THREAD_SAFE}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 *
 */
public interface LongCounter
{
  /**
   * Interface for counter creation.
   */
  interface Creator
  {
    /**
     * Create a new counter.
     * @return new counter
     */
    @NotNull
    LongCounter create();
  }

  /**
   * Simple counter creator.
   * @see SimpleCounter
   */
  Creator SIMPLE = SimpleLongCounter::new;

  /**
   * Thread safe counter creator.
   * @see ThreadSafeCounter
   */
  Creator THREAD_SAFE = ThreadSafeLongCounter::new;

  /**
   *  Add something to the counter.
   *  @param  add  value to add
   *  @return counter value
   */
  long add(long add);

  /**
   *  Add something to the counter.
   *  @param counter counter which value is added
   *  @return counter value
   */
  long add(@NotNull LongCounter counter);

  /**
   * Add a number to the counter.
   * @param number number to add, treated as a long value
   * @return new counter value
   */
  default long add(@NotNull Number number)
  {
    return add(number.longValue());
  }

  /**
   * Add the value of an integer counter.
   * @param counter integer counter
   * @return new counter value
   */
  default long add(@NotNull Counter counter)
  {
    return add(counter.getValue());
  }

  /**
   *  Add 1 to counter.
   *  @return counter value
   */
  long add1();

  /**
   *  Subtract something from the counter.
   *  @param  sub  value to subtract
   *  @return counter value
   */
  long subtract(long sub);

  /**
   *  Subtract something from the counter.
   *  @param counter counter which value is subtracted
   *  @return counter value
   */
  long subtract(@NotNull LongCounter counter);

  /**
   * Subtract a number to the counter.
   * @param number number to add, treated as a long value
   * @return new counter value
   */
  default long subtract(@NotNull Number number)
  {
    return subtract(number.longValue());
  }

  /**
   * Subtract the value of an integer counter.
   * @param counter integer counter
   * @return new counter value
   */
  default long subtract(@NotNull Counter counter)
  {
    return subtract(counter.getValue());
  }

  /**
   *  Subtract 1 from counter.
   *  @return counter value
   */
  long subtract1();

  /**
   *  Get the value.
   *  @return  accumulated value
   */
  long getValue();

  /**
   * Set the value.
   * @param value new value
   */
  void setValue(long value);
}
