// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.util;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

/**
 * A reference to an object.
 * This is similar to {@link java.lang.ref.Reference},
 * but implemented as an interface to also allow hard
 * references and more.
 *
 * @param <T> type of referenced object
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
@FunctionalInterface
public interface IReference<T>
{
  /**
   * Null reference.
   * This always returns {@code null} on {@link #get()}.
   * Use {@link #nullRef()} instead to avoid compiler warnings.
   */
  IReference<?> NULL_REF = new IReference<Object>()
  {
    @Nullable
    @Override
    public Object get()
    {
      return null;
    }

    @Override
    public boolean isNull()
    {
      return true;
    }
  };

  /**
   * Get the referenced object.
   * Depending on the implementation this might return {@code null} if the referenced
   * object is garbage collected or this is a {@link #nullRef() null reference}.
   * @return referenced object, or {@code null}
   */
  @Nullable
  T get();

  /**
   * Is this the null reference?
   * Don't override this method.
   * @return {@code true}: this is the {@link #nullRef() null reference},
   *                       so returning {@code null} is its purpose<br>
   *         {@code false}: this is not the null reference,
   *                        so returning {@code null} means the original object
   *                        is garbage collected and gone to object nirvana
   */
  default boolean isNull()
  {
    return this == NULL_REF;
  }

  /**
   * Get a null reference.
   * @param <R> referenced type
   * @return null reference which is always returning {@code null}
   */
  @SuppressWarnings("unchecked")  // as NULL_REF is independent of type and immutable
  @NotNull
  static <R> IReference<R> nullRef()
  {
    return (IReference<R>)NULL_REF;
  }

  /**
   * Get a hard reference.
   * @param obj object to be referenced
   * @param <R> referenced type
   * @return hard reference
   */
  @NotNull
  static <R> IReference<R> hard(@Nullable R obj)
  {
    return obj == null
            ? nullRef()
            : new HardReferenceImpl<>(obj);
  }

  /**
   * Get a soft reference.
   * This returns a reference which keeps the referenced
   * object via a {@link java.lang.ref.SoftReference}.
   * @param obj object to be referenced
   * @param <R> referenced type
   * @return soft reference
   */
  @NotNull
  static <R> IReference<R> soft(@Nullable R obj)
  {
    return obj == null
            ? nullRef()
            : new SoftReferenceImpl<>(obj);
  }

  /**
   * Get a weak reference.
   * This returns a reference which keeps the referenced
   * object via a {@link java.lang.ref.WeakReference}.
   * @param obj object to be referenced
   * @param <R> referenced type
   * @return weak reference
   */
  @NotNull
  static <R> IReference<R> weak(@Nullable R obj)
  {
    return obj == null
            ? nullRef()
            : new WeakReferenceImpl<>(obj);
  }
}
