// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.util;

import de.caff.annotation.NotNull;

import java.util.function.Supplier;

/**
 * A counter is basically an integer which can be used for counting.
 * <p>
 * Depending on the usage it comes in a
 * {@link ThreadSafeCounter thread-safe implementation}
 * and a {@link SimpleCounter non thread-safe implementation}.
 * <p>
 * There are also factories for both counters:
 * {@link Counter#SIMPLE} and {@link Counter#THREAD_SAFE}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 *
 */
public interface Counter
{
  /**
   * Interface for counter creation.
   * @deprecated outdated, use {@code Supplier<Counter>} instead
   */
  @FunctionalInterface
  @Deprecated
  interface Creator
    extends Supplier<Counter>
  {
    @Override
    default Counter get()
    {
      return create();
    }

    /**
     * Create a new counter.
     * @return new counter
     */
    @NotNull
    Counter create();
  }

  /**
   * Simple counter creator.
   * Instead of using this constant directly it is recommended to
   * use {@link #simple()}.
   * @see SimpleCounter
   */
  Supplier<Counter> SIMPLE = SimpleCounter::new;

  /**
   * Thread safe counter creator.
   * Instead of using this constant directly it is recommended to
   * use {@link #threadSafe()}.
   * @see ThreadSafeCounter
   */
  Supplier<Counter> THREAD_SAFE = ThreadSafeCounter::new;

  /**
   *  Add something to the counter.
   *  @param  add  value to add
   *  @return new counter value
   */
  int add(int add);

  /**
   *  Add something to the counter.
   *  @param counter counter which value is added
   *  @return new counter value
   */
  int add(@NotNull Counter counter);

  /**
   * Add a number to the counter.
   * @param number number to add, treated as an integer value
   * @return new counter value
   */
  default int add(@NotNull Number number)
  {
    return add(number.intValue());
  }

  /**
   *  Add 1 to counter.
   *  @return new counter value
   */
  int add1();

  /**
   *  Subtract something from the counter.
   *  @param  sub  value to subtract
   *  @return counter value
   */
  int subtract(int sub);

  /**
   * Subtract something from the counter.
   * @param sub value to subtract
   * @return counter value
   * @deprecated bad English, use {@link #subtract(int)}
   */
  @Deprecated
  default int substract(int sub)
  {
    return subtract(sub);
  }

  /**
   *  Subtract something from the counter.
   *  @param counter counter which value is subtracted
   *  @return counter value
   */
  int subtract(@NotNull Counter counter);

  /**
   *  Subtract something from the counter.
   *  @param counter counter which value is subtracted
   *  @return counter value
   * @deprecated bad English, use {@link #subtract(Counter)}
   */
  @Deprecated
  default int substract(@NotNull Counter counter)
  {
    return subtract(counter);
  }

  /**
   * Subtract a number to the counter.
   * @param number number to add, treated as an integer value
   * @return new counter value
   */
  default int subtract(@NotNull Number number)
  {
    return subtract(number.intValue());
  }

  /**
   *  Subtract 1 from counter.
   *  @return counter value
   */
  int subtract1();

  /**
   *  Subtract 1 from counter.
   *  @return counter value
   * @deprecated bad English, use {@link #subtract1()}
   */
  @Deprecated
  default int substract1()
  {
    return subtract1();
  }

  /**
   *  Get the value.
   *  @return  accumulated value
   */
  int getValue();

  /**
   * Set the value.
   * @param value new value
   */
  void setValue(int value);

  /**
   * Factory method for creating a simple non-thread-safe counter.
   * @return simple counter with value {@code 0}
   */
  @NotNull
  static Counter simple()
  {
    return SIMPLE.get();
  }

  /**
   * Factory method for creating a simple non-thread-safe counter.
   * @param initial initial value of counter
   * @return simple counter
   */
  @NotNull
  static Counter simple(int initial)
  {
    final Counter counter = simple();
    counter.setValue(initial);
    return counter;
  }

  /**
   * Factory method for creating a thread-safe counter.
   * @return thread-safe counter with value {@code 0}
   */
  @NotNull
  static Counter threadSafe()
  {
    return THREAD_SAFE.get();
  }

  /**
   * Factory method for creating a thread-safe counter.
   * @param initial initial value of counter
   * @return thread-safe counter
   */
  @NotNull
  static Counter threadSafe(int initial)
  {
    final Counter counter = threadSafe();
    counter.setValue(initial);
    return counter;
  }
}
