// ============================================================================
// File:               CopyRef
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2021-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            17.11.21 10:58
//=============================================================================
package de.caff.generics.util;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Copyable;

import java.util.function.UnaryOperator;

/**
 * A reference which copies its object on construction (incoming)
 * and when requested (outgoing).
 * Thus immutabilitly is acquired, as there is no way to change
 * the object kept by this reference even if it is basically mutable.
 * <p>
 * As the object is copied once incoming and each time it is requested
 * this reference is used most efficiently for basically mutable
 * objects which implement an immutable interface. Copying should then return
 * an immutable object which no longer require copying. Then no more than
 * one copy operation has to happen when using this reference.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 17, 2021
 */
public class CopyRef<T extends Copyable<T>>
        implements IReference<T>
{
  @NotNull
  private final UnaryOperator<T> copier;
  @NotNull
  private final T object;

  /**
   * Constructor.
   * @param object object which will be copied into this reference
   * @param copier copier used for copying objects here and on {@link #get}
   */
  public CopyRef(@NotNull T object,
                 @NotNull UnaryOperator<T> copier)
  {
    this.object = object.getCopy();
    this.copier = copier;
  }

  @NotNull
  @Override
  public T get()
  {
    return copier.apply(object);
  }

  /**
   * Create a copy reference for a copyable object.
   * @param object copyable object
   * @param <R> object type
   * @return copy reference
   */
  @NotNull
  public <R extends Copyable<R>> CopyRef<R> make(@NotNull R object)
  {
    return new CopyRef<>(object, R::getCopy);
  }

  /**
   * Create a reference of a copyable object which might be {@code null}.
   * @param object copyable object
   * @param <R> object type
   * @return copy reference or {@link IReference#nullRef() null reference} if object is {@code null}
   */
  @NotNull
  public <R extends Copyable<R>> IReference<R> makeN(@Nullable R object)
  {
    return object == null
            ? IReference.nullRef()
            : make(object);
  }
}
