// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

/**
 * Tuple classes.
 *
 * <p>
 * Tuples provide a way of creating a small group which can hold 2 to 9 objects
 * (called elements here) of different types.
 * Sp Tuples are usually small, and as the arguments are not named in many cases a
 * dedicated class is the better choice. But for temporary usage, e.g. as return type
 * of functions, tuples are quite handy, especially because the functions in the library
 * accept a tuple when the tuple element types are compatible to the function's argument types.
 * </p>
 * <p>
 *   Tuples here come in 3 flavors which are differing whether {@code null} elements are allowed.
 * </p>
 * <ul>
 *   <li>
 *     <p>
 *     <b>ITuple</b>:
 *     Interfaces {@link de.caff.generics.tuple.ITuple2} to {@link de.caff.generics.tuple.ITuple9}
 *     provide the basic access methods, including access with reordering or as tuples of different size.
 *     They don't make any statement about the {@code null}-ness of their elements (meaning that they
 *     obviously could be {@code null}).
 *     </p>
 *     <p>
 *      Both other tuple types provided here implement the appropriate interface. So you can use the interface
 *      in cases where you want to handle both types together. Furthermore these interfaces provide a simple
 *      way to make some of your classes look like a tuple where tuples are required.
 *     </p>
 *   </li>
 *   <li>
 *      <b>Tuple</b>:
 *     The concrete classes {@link de.caff.generics.tuple.Tuple2} to {@link de.caff.generics.tuple.Tuple9}
 *     are small immutable containers used to keep the elements. They only allow non-null elements.
 *   </li>
 *   <li>
 *     <b>NTuple</b>:
 *     The concrete classes {@link de.caff.generics.tuple.NTuple2} to {@link de.caff.generics.tuple.NTuple9}
 *     are small immutable containers used to keep the elements. They allow null elements.
 *   </li>
 * </ul>
 * <h2>Tuple Creation</h2>
 * <p>
 *   The concrete tuples can be created in 3 ways: by constructor, using one of the static
 *   {@code of(...)}/{@code ofNullable(...)} methods of class {@link de.caff.generics.tuple.Tuple}, or
 *   by statically importing the {@code T(...)}/{@code N(...)} methods of class
 *   {@link de.caff.generics.tuple.Tuples} (ordered by decreasing typing).
 * </p>
 * <pre>{@code
 *  t1 = new Tuple3<>(a, b, c);
 *  t2 = Tuple.of(a, b, c);
 *  t3 = T(a, b, c);       // requires import static de.caff.generics.tuple.Tuples.T
 *  assert t1.equals(t2);
 *  assert t2.equals(t3);
 * }</pre>
 * <p>
 *   Recommended are the last two ways, decide according to your preferences.
 * </p>
 * <p>
 *   {@link de.caff.generics.tuple.Tuples} also provides methods for casting a tuple's type in cases
 *   where the compiler is unhappy but you know it shouldn't (see e.g.
 *   {@link de.caff.generics.tuple.Tuples#downCast(de.caff.generics.tuple.Tuple2)}).
 * </p>
 * <h2>Accessing Tuple Elements</h2>
 * <p>
 *   Access to the tuple's elements is done by methods which name consists of an underscore and a number.
 *   The number starts with 1 for the first element (i.e. {@code _1()}. For smaller tuples there are
 *   also accessors which return an reordered tuple which also start with an underscore, followed by the
 *   required order, e.g. in a 3-tuple the method {@code _312()} will return a 3-tuple with the same elements,
 *   but ordered 3,1,2. There is no need for including all elements. For the same 3-tuple the method
 *   {@code _31()} would return a 2-tuple with the elements 3 and 1.
 *  </p>
 *  <p>
 *   For the larger tuples providing these methods would result in a combinatorial catastrophe (i.e. large
 *   classes and long compile times), therefore no reordering on the same level is provided,
 *   and extracting smaller tuples is only provided in order.
 *   E.g. method {@link de.caff.generics.tuple.Tuple9#_13579()} will return a 5-tuple created from
 *   the first, the third, the fifth, the seventh and the ninth element of the original 9-tuple.
 *   But there is no {@code _54321()} method (because there would have to be 15120 of these
 *   &quot;select an arbitrarily ordered 5-tuple&quot; methods).
 * </p>
 * <h2>Tuples and Functions</h2>
 * <p>
 *   Sometimes creating a 2-tuple or 3-tuple is just good enough. E.g. when returning a function
 *   result, and therefore the tuples here provide a way to call a function on them where they
 *   expand themselves to the parameters of the given function, see e.g.
 *   {@link de.caff.generics.tuple.ITuple2#invoke(java.util.function.BiFunction)} or
 *   {@link de.caff.generics.tuple.ITuple3#invoke(de.caff.generics.function.Function3)})}.
 * </p>
 * <p>
 *   The same is true for consumers/procedures
 *   (e.g. {@link de.caff.generics.tuple.ITuple2#sendTo(java.util.function.BiConsumer)} or
 *   {@link de.caff.generics.tuple.ITuple3#sendTo(de.caff.generics.function.Procedure3)} and
 *   predicates (e.g. {@link de.caff.generics.tuple.ITuple2#testBy(java.util.function.BiPredicate)}
 *   of {@link de.caff.generics.tuple.ITuple3#testBy(de.caff.generics.function.Predicate3)}.
 * </p>
 * <p>
 *   2 or more argument function, procedure and predicate implementations defined in this library will also
 *   expose a 1-argument default method which accepts a tuple of correct size and types. Thus a 3-argument
 *   function {@code Function3<R,A,B,C>} will not only provide a method
 * </p>
 * <pre>
 * {@code R apply(A a, B b, C c)}
 * </pre>
 * <p>
 *   but also a method
 * </p>
 * <pre>
 * {@code R apply(@NotNull Tuple<? extends A, ? extends B, ? extends C> tuple)}
 * </pre>
 * <p>
 *   This requires a non-{@code null} argument because the tuple will be expanded inside the method,
 *   but you can make use of {@link de.caff.generics.function.Function1#applyNonNull(java.lang.Object, java.lang.Object)}
 *   in order to achieve this. Standard Java 8 does not have something similar. but you can wrap
 *   a standard Java {@link java.util.function.Function} with
 *   {@link de.caff.generics.function.Function1#from(java.util.function.Function)}.
 *   Similar things (special non-null invoke and wrapping standard Java equivalents) work
 *   with {@link de.caff.generics.function.Procedure1} and  {@link de.caff.generics.function.Predicate1}.
 * </p>
 * <h2>Last Notes</h2>
 * <p>
 *   The tuple classes are created programmatically. It would be awful to write and test all
 *   this combinatorical access methods. The creation code is quite special and not included.
 *   For simple combinatorics helpers see
 *   <a href="{@docRoot}/de/caff/generics/util/combi/package-summary.html#package.description">{@code de.caff.generics.util.combi}</a>.
 * </p>
 */
package de.caff.generics.tuple;