// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.tuple;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

/**
 * Tuple creation and type casting.
 *
 * <p>
 * The tuple factory methods {@code T(...)} (create tuple with non-null elements)
 * and {@code N(...)} (create tuple with nullable elements) are best imported statically.
 * This allows short on-the-fly tuple creation like 
 * </p>
 * <pre>{@code
 *  return T(a, b, c); // returns a 3-tuple (a, b, c)
 * }</pre>
 * <p>
 * instead of more gabby constructor calls.
 * </p>
 * <p>
 * Furthermore, there are {@code cast()} methods which address the problem that Java is
 * overly picky with tuple types because it does not understand that tuples use their types
 * read-only.
 * </p>
 * <p>
 * Assume you have two tuples and want to add then to a more generic collection:
 * </p>
 * <pre>{@code
 * Tuple2<Rectangle2D, Color> coloredRect = T(new Rectangle(), Color.red);
 * Tuple2<Ellipse2D, Color> coloredCircle = T(new Ellipse2D.Double(), Color.green);
 * Collection<Tuple2<Shape, Color>> coloredShapes = new LinkedList<>();
 * coloredShapes.add(coloredRect);   // error: incompatible types: de.caff.generics.tuple.Tuple2<java.awt.geom.Rectangle2D,java.awt.Color> cannot be converted to de.caff.generics.tuple.Tuple2<java.awt.Shape,java.awt.Color>
 * coloredShapes.add(coloredCircle); // error: incompatible types: de.caff.generics.tuple.Tuple2<java.awt.geom.Ellipse2D,java.awt.Color> cannot be converted to de.caff.generics.tuple.Tuple2<java.awt.Shape,java.awt.Color>
 * }</pre>
 * <p>
 * Although both {@code Rectangle} and {@code Ellipse2D} implement {@code Shape} compiling the above will result
 * in the two compiler errors mentioned in the comments.
 * The reason is that the compiler cannot know that it is not possible to set something in a tuple (indeed
 * it could know right now, but cannot know what changes might happen in the future).
 * If it would accept the code for a mutable tuple, somebody could take the {@code coloredShapes}, extract the
 * collection's first tuple, and set its first element to a circle. But the first tuple is the same as
 * {@code coloredRect}, which is convinced that its first element is a rectangle. Ouch!
 * </p>
 * <p>
 * So there is a good reason for Java to be careful, but in this case it is not necessary,
 * as the tuples here are deliberately designed immutable.
 * So to circumvent the problem use the following code instead of the last 2 lines of above example
 * which will make it compile without even a warning (assumes static import of {@code cast}):
 * </p>
 * <pre>{@code
 * coloredShapes.add(cast(coloredRect));
 * coloredShapes.add(cast(coloredCircle));
 * }</pre>
 * <p>
 * All {@code cast()} methods in this class will just directly return their tuple argument as is.
 * The JIT should even be able to remove these calls,
 * </p>
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 07, 2021
 * @see Tuple
 */
public final class Tuples
{
  /** Don't create. */
  private Tuples()
  {
  }

  /**
   * Create a 2-tuple with non-null elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @return 2-tuple with the given elements
   */
  @NotNull
  public static <T1, T2> Tuple2<T1, T2> T(@NotNull T1 e1, @NotNull T2 e2)
  {
    return new Tuple2<>(e1, e2);
  }

  /**
   * Create a 3-tuple with non-null elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @return 3-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3> Tuple3<T1, T2, T3> T(@NotNull T1 e1, @NotNull T2 e2, @NotNull T3 e3)
  {
    return new Tuple3<>(e1, e2, e3);
  }

  /**
   * Create a 4-tuple with non-null elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param e4  fourth tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @param <T4> fourth element type
   * @return 4-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3, T4> Tuple4<T1, T2, T3, T4> T(@NotNull T1 e1, @NotNull T2 e2, @NotNull T3 e3,
                                                          @NotNull T4 e4)
  {
    return new Tuple4<>(e1, e2, e3, e4);
  }

  /**
   * Create a 5-tuple with non-null elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param e4  fourth tuple element
   * @param e5  fifth tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @param <T4> fourth element type
   * @param <T5> fifth element type
   * @return 5-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3, T4, T5>
  Tuple5<T1, T2, T3, T4, T5> T(@NotNull T1 e1, @NotNull T2 e2, @NotNull T3 e3,
                               @NotNull T4 e4, @NotNull T5 e5)
  {
    return new Tuple5<>(e1, e2, e3, e4, e5);
  }

  /**
   * Create a 6-tuple with non-null elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param e4  fourth tuple element
   * @param e5  fifth tuple element
   * @param e6  sixth tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @param <T4> fourth element type
   * @param <T5> fifth element type
   * @param <T6> sixth element type
   * @return 6-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6>
  Tuple6<T1, T2, T3, T4, T5, T6> T(@NotNull T1 e1, @NotNull T2 e2, @NotNull T3 e3,
                                   @NotNull T4 e4, @NotNull T5 e5, @NotNull T6 e6)
  {
    return new Tuple6<>(e1, e2, e3, e4, e5, e6);
  }

  /**
   * Create a 7-tuple with non-null elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param e4  fourth tuple element
   * @param e5  fifth tuple element
   * @param e6  sixth tuple element
   * @param e7  seventh tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @param <T4> fourth element type
   * @param <T5> fifth element type
   * @param <T6> sixth element type
   * @param <T7> seventh element type
   * @return 7-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6, T7>
  Tuple7<T1, T2, T3, T4, T5, T6, T7> T(@NotNull T1 e1, @NotNull T2 e2, @NotNull T3 e3,
                                       @NotNull T4 e4, @NotNull T5 e5, @NotNull T6 e6,
                                       @NotNull T7 e7)
  {
    return new Tuple7<>(e1, e2, e3, e4, e5, e6, e7);
  }

  /**
   * Create an 8-tuple with non-null elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param e4  fourth tuple element
   * @param e5  fifth tuple element
   * @param e6  sixth tuple element
   * @param e7  seventh tuple element
   * @param e8  eighth tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @param <T4> fourth element type
   * @param <T5> fifth element type
   * @param <T6> sixth element type
   * @param <T7> seventh element type
   * @param <T8> eighth element type
   * @return 8-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6, T7, T8>
  Tuple8<T1, T2, T3, T4, T5, T6, T7, T8> T(@NotNull T1 e1, @NotNull T2 e2, @NotNull T3 e3,
                                           @NotNull T4 e4, @NotNull T5 e5, @NotNull T6 e6,
                                           @NotNull T7 e7, @NotNull T8 e8)
  {
    return new Tuple8<>(e1, e2, e3, e4, e5, e6, e7, e8);
  }

  /**
   * Create a 9-tuple with non-null elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param e4  fourth tuple element
   * @param e5  fifth tuple element
   * @param e6  sixth tuple element
   * @param e7  seventh tuple element
   * @param e8  eighth tuple element
   * @param e9  ninth tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @param <T4> fourth element type
   * @param <T5> fifth element type
   * @param <T6> sixth element type
   * @param <T7> seventh element type
   * @param <T8> eighth element type
   * @param <T9> ninth element type
   * @return 9-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6, T7, T8, T9>
  Tuple9<T1, T2, T3, T4, T5, T6, T7, T8, T9> T(@NotNull T1 e1, @NotNull T2 e2, @NotNull T3 e3,
                                               @NotNull T4 e4, @NotNull T5 e5, @NotNull T6 e6,
                                               @NotNull T7 e7, @NotNull T8 e8, @NotNull T9 e9)
  {
    return new Tuple9<>(e1, e2, e3, e4, e5, e6, e7, e8, e9);
  }

  /**
   * Create a 2-tuple with nullable elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @return 2-tuple with the given elements
   */
  @NotNull
  public static <T1, T2> NTuple2<T1, T2> N(@Nullable T1 e1, @Nullable T2 e2)
  {
    return new NTuple2<>(e1, e2);
  }

  /**
   * Create a 3-tuple with nullable elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @return 3-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3> NTuple3<T1, T2, T3> N(@Nullable T1 e1, @Nullable T2 e2, @Nullable T3 e3)
  {
    return new NTuple3<>(e1, e2, e3);
  }

  /**
   * Create a 4-tuple with nullable elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param e4  fourth tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @param <T4> fourth element type
   * @return 4-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3, T4> NTuple4<T1, T2, T3, T4> N(@Nullable T1 e1, @Nullable T2 e2, @Nullable T3 e3,
                                                           @Nullable T4 e4)
  {
    return new NTuple4<>(e1, e2, e3, e4);
  }

  /**
   * Create a 5-tuple with nullable elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param e4  fourth tuple element
   * @param e5  fifth tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @param <T4> fourth element type
   * @param <T5> fifth element type
   * @return 5-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3, T4, T5>
  NTuple5<T1, T2, T3, T4, T5> N(@Nullable T1 e1, @Nullable T2 e2, @Nullable T3 e3,
                                @Nullable T4 e4, @Nullable T5 e5)
  {
    return new NTuple5<>(e1, e2, e3, e4, e5);
  }

  /**
   * Create a 6-tuple with nullable elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param e4  fourth tuple element
   * @param e5  fifth tuple element
   * @param e6  sixth tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @param <T4> fourth element type
   * @param <T5> fifth element type
   * @param <T6> sixth element type
   * @return 6-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6>
  NTuple6<T1, T2, T3, T4, T5, T6> N(@Nullable T1 e1, @Nullable T2 e2, @Nullable T3 e3,
                                    @Nullable T4 e4, @Nullable T5 e5, @Nullable T6 e6)
  {
    return new NTuple6<>(e1, e2, e3, e4, e5, e6);
  }

  /**
   * Create a 7-tuple with nullable elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param e4  fourth tuple element
   * @param e5  fifth tuple element
   * @param e6  sixth tuple element
   * @param e7  seventh tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @param <T4> fourth element type
   * @param <T5> fifth element type
   * @param <T6> sixth element type
   * @param <T7> seventh element type
   * @return 7-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6, T7>
  NTuple7<T1, T2, T3, T4, T5, T6, T7> N(@Nullable T1 e1, @Nullable T2 e2, @Nullable T3 e3,
                                        @Nullable T4 e4, @Nullable T5 e5, @Nullable T6 e6,
                                        @Nullable T7 e7)
  {
    return new NTuple7<>(e1, e2, e3, e4, e5, e6, e7);
  }

  /**
   * Create an 8-tuple with nullable elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param e4  fourth tuple element
   * @param e5  fifth tuple element
   * @param e6  sixth tuple element
   * @param e7  seventh tuple element
   * @param e8  eighth tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @param <T4> fourth element type
   * @param <T5> fifth element type
   * @param <T6> sixth element type
   * @param <T7> seventh element type
   * @param <T8> eighth element type
   * @return 8-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6, T7, T8>
  NTuple8<T1, T2, T3, T4, T5, T6, T7, T8> N(@Nullable T1 e1, @Nullable T2 e2, @Nullable T3 e3,
                                            @Nullable T4 e4, @Nullable T5 e5, @Nullable T6 e6,
                                            @Nullable T7 e7, @Nullable T8 e8)
  {
    return new NTuple8<>(e1, e2, e3, e4, e5, e6, e7, e8);
  }

  /**
   * Create a 9-tuple with nullable elements.
   * @param e1  first tuple element  
   * @param e2  second tuple element
   * @param e3  third tuple element
   * @param e4  fourth tuple element
   * @param e5  fifth tuple element
   * @param e6  sixth tuple element
   * @param e7  seventh tuple element
   * @param e8  eighth tuple element
   * @param e9  ninth tuple element
   * @param <T1> first element type
   * @param <T2> second element type
   * @param <T3> third element type
   * @param <T4> fourth element type
   * @param <T5> fifth element type
   * @param <T6> sixth element type
   * @param <T7> seventh element type
   * @param <T8> eighth element type
   * @param <T9> ninth element type
   * @return 9-tuple with the given elements
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6, T7, T8, T9>
  NTuple9<T1, T2, T3, T4, T5, T6, T7, T8, T9> N(@Nullable T1 e1, @Nullable T2 e2, @Nullable T3 e3,
                                                @Nullable T4 e4, @Nullable T5 e5, @Nullable T6 e6,
                                                @Nullable T7 e7, @Nullable T8 e8, @Nullable T9 e9)
  {
    return new NTuple9<>(e1, e2, e3, e4, e5, e6, e7, e8, e9);
  }

  /**
   * Allow to view a 2-tuple as one with different more basic element types.
   * This will not copy the tuple, this method is mostly useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required last element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as Tuple2 is immutable
  public static <T1, T2> Tuple2<T1, T2> downCast(@NotNull Tuple2<? extends T1, ? extends T2> tuple)
  {
    return (Tuple2<T1, T2>)tuple;
  }  

  /**
   * Allow to view a 3-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as Tuple3 is immutable
  public static <T1, T2, T3> Tuple3<T1, T2, T3> downCast(@NotNull Tuple3<? extends T1, ? extends T2, ? extends T3> tuple)
  {
    return (Tuple3<T1, T2, T3>)tuple;
  }  

  /**
   * Allow to view a 4-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as Tuple4 is immutable
  public static <T1, T2, T3, T4> Tuple4<T1, T2, T3, T4> downCast(@NotNull Tuple4<? extends T1, ? extends T2, ? extends T3, ? extends T4> tuple)
  {
    return (Tuple4<T1, T2, T3, T4>)tuple;
  }  

  /**
   * Allow to view a 5-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as Tuple5 is immutable
  public static <T1, T2, T3, T4, T5> Tuple5<T1, T2, T3, T4, T5>
  downCast(@NotNull Tuple5<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5> tuple)
  {
    return (Tuple5<T1, T2, T3, T4, T5>)tuple;
  }  

  /**
   * Allow to view a 6-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @param <T6> required sixth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as Tuple6 is immutable
  public static <T1, T2, T3, T4, T5, T6> Tuple6<T1, T2, T3, T4, T5, T6>
  downCast(@NotNull Tuple6<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6> tuple)
  {
    return (Tuple6<T1, T2, T3, T4, T5, T6>)tuple;
  }  

  /**
   * Allow to view a 7-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @param <T6> required sixth element type 
   * @param <T7> required seventh element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as Tuple7 is immutable
  public static <T1, T2, T3, T4, T5, T6, T7> Tuple7<T1, T2, T3, T4, T5, T6, T7>
  downCast(@NotNull Tuple7<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7> tuple)
  {
    return (Tuple7<T1, T2, T3, T4, T5, T6, T7>)tuple;
  }  

  /**
   * Allow to view an 8-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @param <T6> required sixth element type 
   * @param <T7> required seventh element type 
   * @param <T8> required eighth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as Tuple8 is immutable
  public static <T1, T2, T3, T4, T5, T6, T7, T8> Tuple8<T1, T2, T3, T4, T5, T6, T7, T8>
  downCast(@NotNull Tuple8<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7, ? extends T8> tuple)
  {
    return (Tuple8<T1, T2, T3, T4, T5, T6, T7, T8>)tuple;
  }  

  /**
   * Allow to view a 9-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @param <T6> required sixth element type 
   * @param <T7> required seventh element type 
   * @param <T8> required eighth element type 
   * @param <T9> required ninth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as Tuple9 is immutable
  public static <T1, T2, T3, T4, T5, T6, T7, T8, T9> Tuple9<T1, T2, T3, T4, T5, T6, T7, T8, T9>
  downCast(@NotNull Tuple9<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7, ? extends T8, ? extends T9> tuple)
  {
    return (Tuple9<T1, T2, T3, T4, T5, T6, T7, T8, T9>)tuple;
  }  

  /**
   * Allow to view a 2-tuple as one with different more basic element types.
   * This will not copy the tuple, this method is mostly useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required last element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as NTuple2 is immutable
  public static <T1, T2> NTuple2<T1, T2> downCast(@NotNull NTuple2<? extends T1, ? extends T2> tuple)
  {
    return (NTuple2<T1, T2>)tuple;
  }  

  /**
   * Allow to view a 3-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as NTuple3 is immutable
  public static <T1, T2, T3> NTuple3<T1, T2, T3> downCast(@NotNull NTuple3<? extends T1, ? extends T2, ? extends T3> tuple)
  {
    return (NTuple3<T1, T2, T3>)tuple;
  }  

  /**
   * Allow to view a 4-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as NTuple4 is immutable
  public static <T1, T2, T3, T4> NTuple4<T1, T2, T3, T4> downCast(@NotNull NTuple4<? extends T1, ? extends T2, ? extends T3, ? extends T4> tuple)
  {
    return (NTuple4<T1, T2, T3, T4>)tuple;
  }  

  /**
   * Allow to view a 5-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as NTuple5 is immutable
  public static <T1, T2, T3, T4, T5> NTuple5<T1, T2, T3, T4, T5>
  downCast(@NotNull NTuple5<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5> tuple)
  {
    return (NTuple5<T1, T2, T3, T4, T5>)tuple;
  }  

  /**
   * Allow to view a 6-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @param <T6> required sixth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as NTuple6 is immutable
  public static <T1, T2, T3, T4, T5, T6> NTuple6<T1, T2, T3, T4, T5, T6>
  downCast(@NotNull NTuple6<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6> tuple)
  {
    return (NTuple6<T1, T2, T3, T4, T5, T6>)tuple;
  }  

  /**
   * Allow to view a 7-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @param <T6> required sixth element type 
   * @param <T7> required seventh element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as NTuple7 is immutable
  public static <T1, T2, T3, T4, T5, T6, T7> NTuple7<T1, T2, T3, T4, T5, T6, T7>
  downCast(@NotNull NTuple7<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7> tuple)
  {
    return (NTuple7<T1, T2, T3, T4, T5, T6, T7>)tuple;
  }  

  /**
   * Allow to view an 8-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @param <T6> required sixth element type 
   * @param <T7> required seventh element type 
   * @param <T8> required eighth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as NTuple8 is immutable
  public static <T1, T2, T3, T4, T5, T6, T7, T8> NTuple8<T1, T2, T3, T4, T5, T6, T7, T8>
  downCast(@NotNull NTuple8<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7, ? extends T8> tuple)
  {
    return (NTuple8<T1, T2, T3, T4, T5, T6, T7, T8>)tuple;
  }  

  /**
   * Allow to view a 9-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @param <T6> required sixth element type 
   * @param <T7> required seventh element type 
   * @param <T8> required eighth element type 
   * @param <T9> required ninth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as NTuple9 is immutable
  public static <T1, T2, T3, T4, T5, T6, T7, T8, T9> NTuple9<T1, T2, T3, T4, T5, T6, T7, T8, T9>
  downCast(@NotNull NTuple9<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7, ? extends T8, ? extends T9> tuple)
  {
    return (NTuple9<T1, T2, T3, T4, T5, T6, T7, T8, T9>)tuple;
  }  

  /**
   * Allow to view a 2-tuple as one with different more basic element types.
   * This will not copy the tuple, this method is mostly useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required last element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as ITuple2 is immutable
  public static <T1, T2> ITuple2<T1, T2> downCast(@NotNull ITuple2<? extends T1, ? extends T2> tuple)
  {
    return (ITuple2<T1, T2>)tuple;
  }  

  /**
   * Allow to view a 3-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as ITuple3 is immutable
  public static <T1, T2, T3> ITuple3<T1, T2, T3> downCast(@NotNull ITuple3<? extends T1, ? extends T2, ? extends T3> tuple)
  {
    return (ITuple3<T1, T2, T3>)tuple;
  }  

  /**
   * Allow to view a 4-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as ITuple4 is immutable
  public static <T1, T2, T3, T4> ITuple4<T1, T2, T3, T4> downCast(@NotNull ITuple4<? extends T1, ? extends T2, ? extends T3, ? extends T4> tuple)
  {
    return (ITuple4<T1, T2, T3, T4>)tuple;
  }  

  /**
   * Allow to view a 5-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as ITuple5 is immutable
  public static <T1, T2, T3, T4, T5> ITuple5<T1, T2, T3, T4, T5>
  downCast(@NotNull ITuple5<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5> tuple)
  {
    return (ITuple5<T1, T2, T3, T4, T5>)tuple;
  }  

  /**
   * Allow to view a 6-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @param <T6> required sixth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as ITuple6 is immutable
  public static <T1, T2, T3, T4, T5, T6> ITuple6<T1, T2, T3, T4, T5, T6>
  downCast(@NotNull ITuple6<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6> tuple)
  {
    return (ITuple6<T1, T2, T3, T4, T5, T6>)tuple;
  }  

  /**
   * Allow to view a 7-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @param <T6> required sixth element type 
   * @param <T7> required seventh element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as ITuple7 is immutable
  public static <T1, T2, T3, T4, T5, T6, T7> ITuple7<T1, T2, T3, T4, T5, T6, T7>
  downCast(@NotNull ITuple7<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7> tuple)
  {
    return (ITuple7<T1, T2, T3, T4, T5, T6, T7>)tuple;
  }  

  /**
   * Allow to view an 8-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @param <T6> required sixth element type 
   * @param <T7> required seventh element type 
   * @param <T8> required eighth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as ITuple8 is immutable
  public static <T1, T2, T3, T4, T5, T6, T7, T8> ITuple8<T1, T2, T3, T4, T5, T6, T7, T8>
  downCast(@NotNull ITuple8<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7, ? extends T8> tuple)
  {
    return (ITuple8<T1, T2, T3, T4, T5, T6, T7, T8>)tuple;
  }  

  /**
   * Allow to view a 9-tuple as one with different more basic element types.
   * This will not copy the tuple nor create any object, this method is useful to avoid compiler warnings.
   * The JIT should be able to remove it.
   * @param tuple tuple to be viewed, with element types which are extensions of the required types
   * @param <T1> required first element type
   * @param <T2> required second element type 
   * @param <T3> required third element type 
   * @param <T4> required fourth element type 
   * @param <T5> required fifth element type 
   * @param <T6> required sixth element type 
   * @param <T7> required seventh element type 
   * @param <T8> required eighth element type 
   * @param <T9> required ninth element type 
   * @return {@code tuple} with adapted element types
   */
  @NotNull
  @SuppressWarnings("unchecked") // as ITuple9 is immutable
  public static <T1, T2, T3, T4, T5, T6, T7, T8, T9> ITuple9<T1, T2, T3, T4, T5, T6, T7, T8, T9>
  downCast(@NotNull ITuple9<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7, ? extends T8, ? extends T9> tuple)
  {
    return (ITuple9<T1, T2, T3, T4, T5, T6, T7, T8, T9>)tuple;
  }
}
