// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021 - 2022  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.tuple;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.io.Serializable;
import java.util.Objects;

/**
 * Immutable tuple with 3 non-null elements.
 *
 * If you want to iterate over a tuple or access its elements by index
 * (note that both is only possible for a common super type of the tuple's types)
 * {@link de.caff.generics.Indexable#viewTuple(ITuple3)} will come to help.
 *
 * Note that this class is automatically created by {@code tools.TupleClassCreator} (not yet public).
 *
 * @param <T1> type of first element
 * @param <T2> type of second element
 * @param <T3> type of third element
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 17, 2021
 * @see NTuple3
 */
public class Tuple3<T1, T2, T3>
        extends ITuple3.Base<T1, T2, T3>
        implements Serializable
{
  private static final long serialVersionUID = 1L;

  @NotNull
  private final T1 elem1;
  @NotNull
  private final T2 elem2;
  @NotNull
  private final T3 elem3;

  /**
   * Constructor.
   * @param e1 first element
   * @param e2 second element
   * @param e3 third element
   */
  public Tuple3(@NotNull T1 e1, @NotNull T2 e2, @NotNull T3 e3)
  {
    assert e1 != null;
    elem1 = e1;
    assert e2 != null;
    elem2 = e2;
    assert e3 != null;
    elem3 = e3;
  }

  @NotNull
  @Override
  public T1 _1()  {
    return elem1;
  }

  @NotNull
  @Override
  public T2 _2()  {
    return elem2;
  }

  @NotNull
  @Override
  public T3 _3()  {
    return elem3;
  }

  @NotNull
  @Override
  public Tuple3<T1, T3, T2> _132()  {
    return new Tuple3<>(elem1, elem3, elem2);
  }

  @NotNull
  @Override
  public Tuple3<T3, T1, T2> _312()  {
    return new Tuple3<>(elem3, elem1, elem2);
  }

  @NotNull
  @Override
  public Tuple3<T3, T2, T1> _321()  {
    return new Tuple3<>(elem3, elem2, elem1);
  }

  @NotNull
  @Override
  public Tuple3<T2, T3, T1> _231()  {
    return new Tuple3<>(elem2, elem3, elem1);
  }

  @NotNull
  @Override
  public Tuple3<T2, T1, T3> _213()  {
    return new Tuple3<>(elem2, elem1, elem3);
  }

  @NotNull
  @Override
  public Tuple2<T1, T2> _12()  {
    return new Tuple2<>(elem1, elem2);
  }

  @NotNull
  @Override
  public Tuple2<T2, T1> _21()  {
    return new Tuple2<>(elem2, elem1);
  }

  @NotNull
  @Override
  public Tuple2<T1, T3> _13()  {
    return new Tuple2<>(elem1, elem3);
  }

  @NotNull
  @Override
  public Tuple2<T3, T1> _31()  {
    return new Tuple2<>(elem3, elem1);
  }

  @NotNull
  @Override
  public Tuple2<T2, T3> _23()  {
    return new Tuple2<>(elem2, elem3);
  }

  @NotNull
  @Override
  public Tuple2<T3, T2> _32()  {
    return new Tuple2<>(elem3, elem2);
  }

  @NotNull
  @Override
  public Tuple3<T1, T2, T3> frozenNotNull()
  {
    return this;
  }

  /**
   * Create a 3-tuple from this one where the first element is exchanged. 
   * @param e1 new first element
   * @return tuple with exchanged element
   * @param <E1> type of exchanged first element
   */
  @NotNull
  public <E1> Tuple3<E1, T2, T3> x1(@NotNull E1 e1)  {
    return new Tuple3<>(e1, elem2, elem3);
  }

  /**
   * Create a 3-tuple from this one where the second element is exchanged. 
   * @param e2 new second element
   * @return tuple with exchanged element
   * @param <E2> type of exchanged second element
   */
  @NotNull
  public <E2> Tuple3<T1, E2, T3> x2(@NotNull E2 e2)  {
    return new Tuple3<>(elem1, e2, elem3);
  }

  /**
   * Create a 3-tuple from this one where the third element is exchanged. 
   * @param e3 new third element
   * @return tuple with exchanged element
   * @param <E3> type of exchanged third element
   */
  @NotNull
  public <E3> Tuple3<T1, T2, E3> x3(@NotNull E3 e3)  {
    return new Tuple3<>(elem1, elem2, e3);
  }

  /**
   * Create a 3-tuple from this one where the first and second elements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   */
  @NotNull
  public <E1, E2> Tuple3<E1, E2, T3> x12(@NotNull E1 e1, @NotNull E2 e2)  {
    return new Tuple3<>(e1, e2, elem3);
  }

  /**
   * Create a 3-tuple from this one where the first and third elements are exchanged. 
   * @param e1 new first element
   * @param e3 new third element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E3> type of exchanged third element
   */
  @NotNull
  public <E1, E3> Tuple3<E1, T2, E3> x13(@NotNull E1 e1, @NotNull E3 e3)  {
    return new Tuple3<>(e1, elem2, e3);
  }

  /**
   * Create a 3-tuple from this one where the second and third elements are exchanged. 
   * @param e2 new second element
   * @param e3 new third element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   */
  @NotNull
  public <E2, E3> Tuple3<T1, E2, E3> x23(@NotNull E2 e2, @NotNull E3 e3)  {
    return new Tuple3<>(elem1, e2, e3);
  }

  /**
   * Create a 4-tuple from this 3-tuple by inserting a new first element.
   * @param e1 new element
   * @param <T> new element type
   * @return 4-tuple
   */
  @NotNull
  public <T> Tuple4<T, T1, T2, T3> i1(@NotNull T e1)
  {
    return new Tuple4<>(e1, elem1, elem2, elem3);
  }

  /**
   * Create a 4-tuple from this 3-tuple by inserting a new second element.
   * @param e2 new element
   * @param <T> new element type
   * @return 4-tuple
   */
  @NotNull
  public <T> Tuple4<T1, T, T2, T3> i2(@NotNull T e2)
  {
    return new Tuple4<>(elem1, e2, elem2, elem3);
  }

  /**
   * Create a 4-tuple from this 3-tuple by inserting a new third element.
   * @param e3 new element
   * @param <T> new element type
   * @return 4-tuple
   */
  @NotNull
  public <T> Tuple4<T1, T2, T, T3> i3(@NotNull T e3)
  {
    return new Tuple4<>(elem1, elem2, e3, elem3);
  }

  /**
   * Create a 4-tuple from this 3-tuple by adding a new fourth element.
   * @param e4 new element
   * @param <T> new element type
   * @return 4-tuple
   */
  @NotNull
  public <T> Tuple4<T1, T2, T3, T> i4(@NotNull T e4)
  {
    return new Tuple4<>(elem1, elem2, elem3, e4);
  }

  @NotNull
  @Override
  public String toString()
  {
    return ITuple3.toString("Tuple3", this);
  }

}
