// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021 - 2022  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.tuple;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.io.Serializable;
import java.util.Objects;

/**
 * Immutable tuple with 2 non-null elements.
 *
 * If you want to iterate over a tuple or access its elements by index
 * (note that both is only possible for a common super type of the tuple's types)
 * {@link de.caff.generics.Indexable#viewTuple(ITuple2)} will come to help.
 *
 * Note that this class is automatically created by {@code tools.TupleClassCreator} (not yet public).
 *
 * @param <T1> type of first element
 * @param <T2> type of second element
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 17, 2021
 * @see NTuple2
 */
public class Tuple2<T1, T2>
        extends ITuple2.Base<T1, T2>
        implements Serializable
{
  private static final long serialVersionUID = 1L;

  @NotNull
  private final T1 elem1;
  @NotNull
  private final T2 elem2;

  /**
   * Constructor.
   * @param e1 first element
   * @param e2 second element
   */
  public Tuple2(@NotNull T1 e1, @NotNull T2 e2)
  {
    assert e1 != null;
    elem1 = e1;
    assert e2 != null;
    elem2 = e2;
  }

  @NotNull
  @Override
  public T1 _1()  {
    return elem1;
  }

  @NotNull
  @Override
  public T2 _2()  {
    return elem2;
  }

  @NotNull
  @Override
  public Tuple2<T2, T1> _21()  {
    return new Tuple2<>(elem2, elem1);
  }

  @NotNull
  @Override
  public Tuple2<T1, T2> frozenNotNull()
  {
    return this;
  }

  /**
   * Create a 2-tuple from this one where the first element is exchanged. 
   * @param e1 new first element
   * @return tuple with exchanged element
   * @param <E1> type of exchanged first element
   */
  @NotNull
  public <E1> Tuple2<E1, T2> x1(@NotNull E1 e1)  {
    return new Tuple2<>(e1, elem2);
  }

  /**
   * Create a 2-tuple from this one where the second element is exchanged. 
   * @param e2 new second element
   * @return tuple with exchanged element
   * @param <E2> type of exchanged second element
   */
  @NotNull
  public <E2> Tuple2<T1, E2> x2(@NotNull E2 e2)  {
    return new Tuple2<>(elem1, e2);
  }

  /**
   * Create a 3-tuple from this 2-tuple by inserting a new first element.
   * @param e1 new element
   * @param <T> new element type
   * @return 3-tuple
   */
  @NotNull
  public <T> Tuple3<T, T1, T2> i1(@NotNull T e1)
  {
    return new Tuple3<>(e1, elem1, elem2);
  }

  /**
   * Create a 3-tuple from this 2-tuple by inserting a new second element.
   * @param e2 new element
   * @param <T> new element type
   * @return 3-tuple
   */
  @NotNull
  public <T> Tuple3<T1, T, T2> i2(@NotNull T e2)
  {
    return new Tuple3<>(elem1, e2, elem2);
  }

  /**
   * Create a 3-tuple from this 2-tuple by adding a new third element.
   * @param e3 new element
   * @param <T> new element type
   * @return 3-tuple
   */
  @NotNull
  public <T> Tuple3<T1, T2, T> i3(@NotNull T e3)
  {
    return new Tuple3<>(elem1, elem2, e3);
  }

  @NotNull
  @Override
  public String toString()
  {
    return ITuple2.toString("Tuple2", this);
  }

}
