// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.tuple;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

/**
 * Tuple helper class.
 * This provides the {@code of(...)} and {@code ofNullable(...)} tuple creation methods which are a bit
 * more wordy versions of the {@code T(...)} and {@code N(...)} methods of {@link Tuples}.
 * <p>
 * So both
 * <pre>{@code
 * Tuple.of(a, b) // from this class
 * T(a, b),  // requires import static Tuples.T;
 * }</pre>
 * will return equal 2-tuples.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 18, 2021
 */
public final class Tuple
{
  /** Don't create. */
  private Tuple()
  {}

  /**
   * Create a tuple with two non-null values.
   * @param value1 first value
   * @param value2 second value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @return 2-tuple
   */
  @NotNull
  public static <T1, T2> Tuple2<T1, T2> of(@NotNull T1 value1,
                                           @NotNull T2 value2)
  {
    return new Tuple2<>(value1, value2);
  }

  /**
   * Create a tuple with two values which can possibly be null.
   * @param value1 first value
   * @param value2 second value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @return 2-tuple
   */
  @NotNull
  public static <T1, T2> NTuple2<T1, T2> ofNullable(@Nullable T1 value1,
                                                    @Nullable T2 value2)
  {
    return new NTuple2<>(value1, value2);
  }

  /**
   * Create a tuple with 3 non-null values.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @return 3-tuple
   */
  @NotNull
  public static <T1, T2, T3> Tuple3<T1, T2, T3> of(@NotNull T1 value1,
                                                   @NotNull T2 value2,
                                                   @NotNull T3 value3)
  {
    return new Tuple3<>(value1, value2, value3);
  }

  /**
   * Create a tuple with 3 values which can possibly be null.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @return 3-tuple
   */
  @NotNull
  public static <T1, T2, T3> NTuple3<T1, T2, T3> ofNullable(@Nullable T1 value1,
                                                            @Nullable T2 value2,
                                                            @Nullable T3 value3)
  {
    return new NTuple3<>(value1, value2, value3);
  }

  /**
   * Create a tuple with 4 non-null values.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param value4 fourth value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @param <T4> type of fourth value
   * @return 4-tuple
   */
  @NotNull
  public static <T1, T2, T3, T4>
  Tuple4<T1, T2, T3, T4> of(@NotNull T1 value1,
                            @NotNull T2 value2,
                            @NotNull T3 value3,
                            @NotNull T4 value4)
  {
    return new Tuple4<>(value1, value2, value3, value4);
  }

  /**
   * Create a tuple with 4 values which can possibly be null.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param value4 fourth value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @param <T4> type of fourth value
   * @return 4-tuple
   */
  @NotNull
  public static <T1, T2, T3, T4>
  NTuple4<T1, T2, T3, T4>  ofNullable(@Nullable T1 value1,
                                      @Nullable T2 value2,
                                      @Nullable T3 value3,
                                      @Nullable T4 value4)
  {
    return new NTuple4<>(value1, value2, value3, value4);
  }

  /**
   * Create a tuple with 5 non-null values.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param value4 fourth value
   * @param value5 fifth value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @param <T4> type of fourth value
   * @param <T5> type of fifth value
   * @return 5-tuple
   */
  @NotNull
  public static <T1, T2, T3, T4, T5>
  Tuple5<T1, T2, T3, T4, T5> of(@NotNull T1 value1,
                                @NotNull T2 value2,
                                @NotNull T3 value3,
                                @NotNull T4 value4,
                                @NotNull T5 value5)
  {
    return new Tuple5<>(value1, value2, value3, value4, value5);
  }

  /**
   * Create a tuple with 5 values which can possibly be null.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param value4 fourth value
   * @param value5 fifth value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @param <T4> type of fourth value
   * @param <T5> type of fifth value
   * @return 5-tuple
   */
  @NotNull
  public static <T1, T2, T3, T4, T5>
  NTuple5<T1, T2, T3, T4, T5> ofNullable(@Nullable T1 value1,
                                         @Nullable T2 value2,
                                         @Nullable T3 value3,
                                         @Nullable T4 value4,
                                         @Nullable T5 value5)
  {
    return new NTuple5<>(value1, value2, value3, value4, value5);
  }

  /**
   * Create a tuple with 6 non-null values.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param value4 fourth value
   * @param value5 fifth value
   * @param value6 sixth value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @param <T4> type of fourth value
   * @param <T5> type of fifth value
   * @param <T6> type of sixth value
   * @return 6-tuple
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6>
  Tuple6<T1, T2, T3, T4, T5, T6> of(@NotNull T1 value1,
                                    @NotNull T2 value2,
                                    @NotNull T3 value3,
                                    @NotNull T4 value4,
                                    @NotNull T5 value5,
                                    @NotNull T6 value6)
  {
    return new Tuple6<>(value1, value2, value3, value4, value5, value6);
  }

  /**
   * Create a tuple with 6 values which can possibly be null.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param value4 fourth value
   * @param value5 fifth value
   * @param value6 sixth value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @param <T4> type of fourth value
   * @param <T5> type of fifth value
   * @param <T6> type of sixth value
   * @return 6-tuple
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6>
  NTuple6<T1, T2, T3, T4, T5, T6> ofNullable(@Nullable T1 value1,
                                             @Nullable T2 value2,
                                             @Nullable T3 value3,
                                             @Nullable T4 value4,
                                             @Nullable T5 value5,
                                             @Nullable T6 value6)
  {
    return new NTuple6<>(value1, value2, value3, value4, value5, value6);
  }

  /**
   * Create a tuple with 7 non-null values.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param value4 fourth value
   * @param value5 fifth value
   * @param value6 sixth value
   * @param value7 seventh value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @param <T4> type of fourth value
   * @param <T5> type of fifth value
   * @param <T6> type of sixth value
   * @param <T7> type of seventh value
   * @return 7-tuple
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6, T7>
  Tuple7<T1, T2, T3, T4, T5, T6, T7> of(@NotNull T1 value1,
                                        @NotNull T2 value2,
                                        @NotNull T3 value3,
                                        @NotNull T4 value4,
                                        @NotNull T5 value5,
                                        @NotNull T6 value6,
                                        @NotNull T7 value7)
  {
    return new Tuple7<>(value1, value2, value3, value4, value5, value6, value7);
  }

  /**
   * Create a tuple with 7 values which can possibly be null.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param value4 fourth value
   * @param value5 fifth value
   * @param value6 sixth value
   * @param value7 seventh value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @param <T4> type of fourth value
   * @param <T5> type of fifth value
   * @param <T6> type of sixth value
   * @param <T7> type of seventh value
   * @return 7-tuple
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6, T7>
  NTuple7<T1, T2, T3, T4, T5, T6, T7> ofNullable(@Nullable T1 value1,
                                                 @Nullable T2 value2,
                                                 @Nullable T3 value3,
                                                 @Nullable T4 value4,
                                                 @Nullable T5 value5,
                                                 @Nullable T6 value6,
                                                 @Nullable T7 value7)
  {
    return new NTuple7<>(value1, value2, value3, value4, value5, value6, value7);
  }

  /**
   * Create a tuple with 8 non-null values.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param value4 fourth value
   * @param value5 fifth value
   * @param value6 sixth value
   * @param value7 seventh value
   * @param value8 eighth value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @param <T4> type of fourth value
   * @param <T5> type of fifth value
   * @param <T6> type of sixth value
   * @param <T7> type of seventh value
   * @param <T8> type of eighth value
   * @return 8-tuple
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6, T7, T8>
  Tuple8<T1, T2, T3, T4, T5, T6, T7, T8> of(@NotNull T1 value1,
                                            @NotNull T2 value2,
                                            @NotNull T3 value3,
                                            @NotNull T4 value4,
                                            @NotNull T5 value5,
                                            @NotNull T6 value6,
                                            @NotNull T7 value7,
                                            @NotNull T8 value8)
  {
    return new Tuple8<>(value1, value2, value3, value4, value5, value6, value7, value8);
  }

  /**
   * Create a tuple with 8 values which can possibly be null.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param value4 fourth value
   * @param value5 fifth value
   * @param value6 sixth value
   * @param value7 seventh value
   * @param value8 eighth value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @param <T4> type of fourth value
   * @param <T5> type of fifth value
   * @param <T6> type of sixth value
   * @param <T7> type of seventh value
   * @param <T8> type of eighth value
   * @return 8-tuple
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6, T7, T8>
  NTuple8<T1, T2, T3, T4, T5, T6, T7, T8> ofNullable(@Nullable T1 value1,
                                                     @Nullable T2 value2,
                                                     @Nullable T3 value3,
                                                     @Nullable T4 value4,
                                                     @Nullable T5 value5,
                                                     @Nullable T6 value6,
                                                     @Nullable T7 value7,
                                                     @Nullable T8 value8)
  {
    return new NTuple8<>(value1, value2, value3, value4, value5, value6, value7, value8);
  }

  /**
   * Create a tuple with 9 non-null values.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param value4 fourth value
   * @param value5 fifth value
   * @param value6 sixth value
   * @param value7 seventh value
   * @param value8 eighth value
   * @param value9 nineth value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @param <T4> type of fourth value
   * @param <T5> type of fifth value
   * @param <T6> type of sixth value
   * @param <T7> type of seventh value
   * @param <T8> type of eighth value
   * @param <T9> type of nineth value
   * @return 9-tuple
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6, T7, T8, T9>
  Tuple9<T1, T2, T3, T4, T5, T6, T7, T8, T9> of(@NotNull T1 value1,
                                                @NotNull T2 value2,
                                                @NotNull T3 value3,
                                                @NotNull T4 value4,
                                                @NotNull T5 value5,
                                                @NotNull T6 value6,
                                                @NotNull T7 value7,
                                                @NotNull T8 value8,
                                                @NotNull T9 value9)
  {
    return new Tuple9<>(value1, value2, value3, value4, value5, value6, value7, value8, value9);
  }

  /**
   * Create a tuple with 9 values which can possibly be null.
   * @param value1 first value
   * @param value2 second value
   * @param value3 third value
   * @param value4 fourth value
   * @param value5 fifth value
   * @param value6 sixth value
   * @param value7 seventh value
   * @param value8 eighth value
   * @param value9 nineth value
   * @param <T1> type of first value
   * @param <T2> type of second value
   * @param <T3> type of third value
   * @param <T4> type of fourth value
   * @param <T5> type of fifth value
   * @param <T6> type of sixth value
   * @param <T7> type of seventh value
   * @param <T8> type of eighth value
   * @param <T9> type of nineth value
   * @return 9-tuple
   */
  @NotNull
  public static <T1, T2, T3, T4, T5, T6, T7, T8, T9>
  NTuple9<T1, T2, T3, T4, T5, T6, T7, T8, T9> ofNullable(@Nullable T1 value1,
                                                         @Nullable T2 value2,
                                                         @Nullable T3 value3,
                                                         @Nullable T4 value4,
                                                         @Nullable T5 value5,
                                                         @Nullable T6 value6,
                                                         @Nullable T7 value7,
                                                         @Nullable T8 value8,
                                                         @Nullable T9 value9)
  {
    return new NTuple9<>(value1, value2, value3, value4, value5, value6, value7, value8, value9);
  }
}
