// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021 - 2022  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.tuple;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.io.Serializable;
import java.util.Objects;

/**
 * Immutable tuple with 6 nullable elements.
 *
 * If you want to iterate over a tuple or access its elements by index
 * (note that both is only possible for a common super type of the tuple's types)
 * {@link de.caff.generics.Indexable#viewTuple(ITuple6)} will come to help.
 *
 * Note that this class is automatically created by {@code tools.TupleClassCreator} (not yet public).
 *
 * @param <T1> type of first element
 * @param <T2> type of second element
 * @param <T3> type of third element
 * @param <T4> type of fourth element
 * @param <T5> type of fifth element
 * @param <T6> type of sixth element
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 17, 2021
 * @see Tuple6
 */
public class NTuple6<T1, T2, T3, T4, T5, T6>
        extends ITuple6.Base<T1, T2, T3, T4, T5, T6>
        implements Serializable
{
  private static final long serialVersionUID = 1L;

  @Nullable
  private final T1 elem1;
  @Nullable
  private final T2 elem2;
  @Nullable
  private final T3 elem3;
  @Nullable
  private final T4 elem4;
  @Nullable
  private final T5 elem5;
  @Nullable
  private final T6 elem6;

  /**
   * Constructor.
   * @param e1 first element
   * @param e2 second element
   * @param e3 third element
   * @param e4 fourth element
   * @param e5 fifth element
   * @param e6 sixth element
   */
  public NTuple6(@Nullable T1 e1, @Nullable T2 e2, @Nullable T3 e3, @Nullable T4 e4, @Nullable T5 e5, @Nullable T6 e6)
  {
    elem1 = e1;
    elem2 = e2;
    elem3 = e3;
    elem4 = e4;
    elem5 = e5;
    elem6 = e6;
  }

  @Nullable
  @Override
  public T1 _1()  {
    return elem1;
  }

  @Nullable
  @Override
  public T2 _2()  {
    return elem2;
  }

  @Nullable
  @Override
  public T3 _3()  {
    return elem3;
  }

  @Nullable
  @Override
  public T4 _4()  {
    return elem4;
  }

  @Nullable
  @Override
  public T5 _5()  {
    return elem5;
  }

  @Nullable
  @Override
  public T6 _6()  {
    return elem6;
  }

  @NotNull
  @Override
  public NTuple5<T1, T2, T3, T4, T5> _12345()  {
    return new NTuple5<>(elem1, elem2, elem3, elem4, elem5);
  }

  @NotNull
  @Override
  public NTuple5<T1, T2, T3, T4, T6> _12346()  {
    return new NTuple5<>(elem1, elem2, elem3, elem4, elem6);
  }

  @NotNull
  @Override
  public NTuple5<T1, T2, T3, T5, T6> _12356()  {
    return new NTuple5<>(elem1, elem2, elem3, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple5<T1, T2, T4, T5, T6> _12456()  {
    return new NTuple5<>(elem1, elem2, elem4, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple5<T1, T3, T4, T5, T6> _13456()  {
    return new NTuple5<>(elem1, elem3, elem4, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple5<T2, T3, T4, T5, T6> _23456()  {
    return new NTuple5<>(elem2, elem3, elem4, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple4<T1, T2, T3, T4> _1234()  {
    return new NTuple4<>(elem1, elem2, elem3, elem4);
  }

  @NotNull
  @Override
  public NTuple4<T1, T2, T3, T5> _1235()  {
    return new NTuple4<>(elem1, elem2, elem3, elem5);
  }

  @NotNull
  @Override
  public NTuple4<T1, T2, T4, T5> _1245()  {
    return new NTuple4<>(elem1, elem2, elem4, elem5);
  }

  @NotNull
  @Override
  public NTuple4<T1, T3, T4, T5> _1345()  {
    return new NTuple4<>(elem1, elem3, elem4, elem5);
  }

  @NotNull
  @Override
  public NTuple4<T2, T3, T4, T5> _2345()  {
    return new NTuple4<>(elem2, elem3, elem4, elem5);
  }

  @NotNull
  @Override
  public NTuple4<T1, T2, T3, T6> _1236()  {
    return new NTuple4<>(elem1, elem2, elem3, elem6);
  }

  @NotNull
  @Override
  public NTuple4<T1, T2, T4, T6> _1246()  {
    return new NTuple4<>(elem1, elem2, elem4, elem6);
  }

  @NotNull
  @Override
  public NTuple4<T1, T3, T4, T6> _1346()  {
    return new NTuple4<>(elem1, elem3, elem4, elem6);
  }

  @NotNull
  @Override
  public NTuple4<T2, T3, T4, T6> _2346()  {
    return new NTuple4<>(elem2, elem3, elem4, elem6);
  }

  @NotNull
  @Override
  public NTuple4<T1, T2, T5, T6> _1256()  {
    return new NTuple4<>(elem1, elem2, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple4<T1, T3, T5, T6> _1356()  {
    return new NTuple4<>(elem1, elem3, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple4<T2, T3, T5, T6> _2356()  {
    return new NTuple4<>(elem2, elem3, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple4<T1, T4, T5, T6> _1456()  {
    return new NTuple4<>(elem1, elem4, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple4<T2, T4, T5, T6> _2456()  {
    return new NTuple4<>(elem2, elem4, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple4<T3, T4, T5, T6> _3456()  {
    return new NTuple4<>(elem3, elem4, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T1, T2, T3> _123()  {
    return new NTuple3<>(elem1, elem2, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T1, T3, T2> _132()  {
    return new NTuple3<>(elem1, elem3, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T3, T1, T2> _312()  {
    return new NTuple3<>(elem3, elem1, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T3, T2, T1> _321()  {
    return new NTuple3<>(elem3, elem2, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T2, T3, T1> _231()  {
    return new NTuple3<>(elem2, elem3, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T2, T1, T3> _213()  {
    return new NTuple3<>(elem2, elem1, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T1, T2, T4> _124()  {
    return new NTuple3<>(elem1, elem2, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T1, T4, T2> _142()  {
    return new NTuple3<>(elem1, elem4, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T4, T1, T2> _412()  {
    return new NTuple3<>(elem4, elem1, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T4, T2, T1> _421()  {
    return new NTuple3<>(elem4, elem2, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T2, T4, T1> _241()  {
    return new NTuple3<>(elem2, elem4, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T2, T1, T4> _214()  {
    return new NTuple3<>(elem2, elem1, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T1, T3, T4> _134()  {
    return new NTuple3<>(elem1, elem3, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T1, T4, T3> _143()  {
    return new NTuple3<>(elem1, elem4, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T4, T1, T3> _413()  {
    return new NTuple3<>(elem4, elem1, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T4, T3, T1> _431()  {
    return new NTuple3<>(elem4, elem3, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T3, T4, T1> _341()  {
    return new NTuple3<>(elem3, elem4, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T3, T1, T4> _314()  {
    return new NTuple3<>(elem3, elem1, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T2, T3, T4> _234()  {
    return new NTuple3<>(elem2, elem3, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T2, T4, T3> _243()  {
    return new NTuple3<>(elem2, elem4, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T4, T2, T3> _423()  {
    return new NTuple3<>(elem4, elem2, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T4, T3, T2> _432()  {
    return new NTuple3<>(elem4, elem3, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T3, T4, T2> _342()  {
    return new NTuple3<>(elem3, elem4, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T3, T2, T4> _324()  {
    return new NTuple3<>(elem3, elem2, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T1, T2, T5> _125()  {
    return new NTuple3<>(elem1, elem2, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T1, T5, T2> _152()  {
    return new NTuple3<>(elem1, elem5, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T5, T1, T2> _512()  {
    return new NTuple3<>(elem5, elem1, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T5, T2, T1> _521()  {
    return new NTuple3<>(elem5, elem2, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T2, T5, T1> _251()  {
    return new NTuple3<>(elem2, elem5, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T2, T1, T5> _215()  {
    return new NTuple3<>(elem2, elem1, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T1, T3, T5> _135()  {
    return new NTuple3<>(elem1, elem3, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T1, T5, T3> _153()  {
    return new NTuple3<>(elem1, elem5, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T5, T1, T3> _513()  {
    return new NTuple3<>(elem5, elem1, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T5, T3, T1> _531()  {
    return new NTuple3<>(elem5, elem3, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T3, T5, T1> _351()  {
    return new NTuple3<>(elem3, elem5, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T3, T1, T5> _315()  {
    return new NTuple3<>(elem3, elem1, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T2, T3, T5> _235()  {
    return new NTuple3<>(elem2, elem3, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T2, T5, T3> _253()  {
    return new NTuple3<>(elem2, elem5, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T5, T2, T3> _523()  {
    return new NTuple3<>(elem5, elem2, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T5, T3, T2> _532()  {
    return new NTuple3<>(elem5, elem3, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T3, T5, T2> _352()  {
    return new NTuple3<>(elem3, elem5, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T3, T2, T5> _325()  {
    return new NTuple3<>(elem3, elem2, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T1, T4, T5> _145()  {
    return new NTuple3<>(elem1, elem4, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T1, T5, T4> _154()  {
    return new NTuple3<>(elem1, elem5, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T5, T1, T4> _514()  {
    return new NTuple3<>(elem5, elem1, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T5, T4, T1> _541()  {
    return new NTuple3<>(elem5, elem4, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T4, T5, T1> _451()  {
    return new NTuple3<>(elem4, elem5, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T4, T1, T5> _415()  {
    return new NTuple3<>(elem4, elem1, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T2, T4, T5> _245()  {
    return new NTuple3<>(elem2, elem4, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T2, T5, T4> _254()  {
    return new NTuple3<>(elem2, elem5, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T5, T2, T4> _524()  {
    return new NTuple3<>(elem5, elem2, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T5, T4, T2> _542()  {
    return new NTuple3<>(elem5, elem4, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T4, T5, T2> _452()  {
    return new NTuple3<>(elem4, elem5, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T4, T2, T5> _425()  {
    return new NTuple3<>(elem4, elem2, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T3, T4, T5> _345()  {
    return new NTuple3<>(elem3, elem4, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T3, T5, T4> _354()  {
    return new NTuple3<>(elem3, elem5, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T5, T3, T4> _534()  {
    return new NTuple3<>(elem5, elem3, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T5, T4, T3> _543()  {
    return new NTuple3<>(elem5, elem4, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T4, T5, T3> _453()  {
    return new NTuple3<>(elem4, elem5, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T4, T3, T5> _435()  {
    return new NTuple3<>(elem4, elem3, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T1, T2, T6> _126()  {
    return new NTuple3<>(elem1, elem2, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T1, T6, T2> _162()  {
    return new NTuple3<>(elem1, elem6, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T6, T1, T2> _612()  {
    return new NTuple3<>(elem6, elem1, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T6, T2, T1> _621()  {
    return new NTuple3<>(elem6, elem2, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T2, T6, T1> _261()  {
    return new NTuple3<>(elem2, elem6, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T2, T1, T6> _216()  {
    return new NTuple3<>(elem2, elem1, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T1, T3, T6> _136()  {
    return new NTuple3<>(elem1, elem3, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T1, T6, T3> _163()  {
    return new NTuple3<>(elem1, elem6, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T6, T1, T3> _613()  {
    return new NTuple3<>(elem6, elem1, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T6, T3, T1> _631()  {
    return new NTuple3<>(elem6, elem3, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T3, T6, T1> _361()  {
    return new NTuple3<>(elem3, elem6, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T3, T1, T6> _316()  {
    return new NTuple3<>(elem3, elem1, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T2, T3, T6> _236()  {
    return new NTuple3<>(elem2, elem3, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T2, T6, T3> _263()  {
    return new NTuple3<>(elem2, elem6, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T6, T2, T3> _623()  {
    return new NTuple3<>(elem6, elem2, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T6, T3, T2> _632()  {
    return new NTuple3<>(elem6, elem3, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T3, T6, T2> _362()  {
    return new NTuple3<>(elem3, elem6, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T3, T2, T6> _326()  {
    return new NTuple3<>(elem3, elem2, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T1, T4, T6> _146()  {
    return new NTuple3<>(elem1, elem4, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T1, T6, T4> _164()  {
    return new NTuple3<>(elem1, elem6, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T6, T1, T4> _614()  {
    return new NTuple3<>(elem6, elem1, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T6, T4, T1> _641()  {
    return new NTuple3<>(elem6, elem4, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T4, T6, T1> _461()  {
    return new NTuple3<>(elem4, elem6, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T4, T1, T6> _416()  {
    return new NTuple3<>(elem4, elem1, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T2, T4, T6> _246()  {
    return new NTuple3<>(elem2, elem4, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T2, T6, T4> _264()  {
    return new NTuple3<>(elem2, elem6, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T6, T2, T4> _624()  {
    return new NTuple3<>(elem6, elem2, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T6, T4, T2> _642()  {
    return new NTuple3<>(elem6, elem4, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T4, T6, T2> _462()  {
    return new NTuple3<>(elem4, elem6, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T4, T2, T6> _426()  {
    return new NTuple3<>(elem4, elem2, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T3, T4, T6> _346()  {
    return new NTuple3<>(elem3, elem4, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T3, T6, T4> _364()  {
    return new NTuple3<>(elem3, elem6, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T6, T3, T4> _634()  {
    return new NTuple3<>(elem6, elem3, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T6, T4, T3> _643()  {
    return new NTuple3<>(elem6, elem4, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T4, T6, T3> _463()  {
    return new NTuple3<>(elem4, elem6, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T4, T3, T6> _436()  {
    return new NTuple3<>(elem4, elem3, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T1, T5, T6> _156()  {
    return new NTuple3<>(elem1, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T1, T6, T5> _165()  {
    return new NTuple3<>(elem1, elem6, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T6, T1, T5> _615()  {
    return new NTuple3<>(elem6, elem1, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T6, T5, T1> _651()  {
    return new NTuple3<>(elem6, elem5, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T5, T6, T1> _561()  {
    return new NTuple3<>(elem5, elem6, elem1);
  }

  @NotNull
  @Override
  public NTuple3<T5, T1, T6> _516()  {
    return new NTuple3<>(elem5, elem1, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T2, T5, T6> _256()  {
    return new NTuple3<>(elem2, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T2, T6, T5> _265()  {
    return new NTuple3<>(elem2, elem6, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T6, T2, T5> _625()  {
    return new NTuple3<>(elem6, elem2, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T6, T5, T2> _652()  {
    return new NTuple3<>(elem6, elem5, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T5, T6, T2> _562()  {
    return new NTuple3<>(elem5, elem6, elem2);
  }

  @NotNull
  @Override
  public NTuple3<T5, T2, T6> _526()  {
    return new NTuple3<>(elem5, elem2, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T3, T5, T6> _356()  {
    return new NTuple3<>(elem3, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T3, T6, T5> _365()  {
    return new NTuple3<>(elem3, elem6, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T6, T3, T5> _635()  {
    return new NTuple3<>(elem6, elem3, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T6, T5, T3> _653()  {
    return new NTuple3<>(elem6, elem5, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T5, T6, T3> _563()  {
    return new NTuple3<>(elem5, elem6, elem3);
  }

  @NotNull
  @Override
  public NTuple3<T5, T3, T6> _536()  {
    return new NTuple3<>(elem5, elem3, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T4, T5, T6> _456()  {
    return new NTuple3<>(elem4, elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple3<T4, T6, T5> _465()  {
    return new NTuple3<>(elem4, elem6, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T6, T4, T5> _645()  {
    return new NTuple3<>(elem6, elem4, elem5);
  }

  @NotNull
  @Override
  public NTuple3<T6, T5, T4> _654()  {
    return new NTuple3<>(elem6, elem5, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T5, T6, T4> _564()  {
    return new NTuple3<>(elem5, elem6, elem4);
  }

  @NotNull
  @Override
  public NTuple3<T5, T4, T6> _546()  {
    return new NTuple3<>(elem5, elem4, elem6);
  }

  @NotNull
  @Override
  public NTuple2<T1, T2> _12()  {
    return new NTuple2<>(elem1, elem2);
  }

  @NotNull
  @Override
  public NTuple2<T2, T1> _21()  {
    return new NTuple2<>(elem2, elem1);
  }

  @NotNull
  @Override
  public NTuple2<T1, T3> _13()  {
    return new NTuple2<>(elem1, elem3);
  }

  @NotNull
  @Override
  public NTuple2<T3, T1> _31()  {
    return new NTuple2<>(elem3, elem1);
  }

  @NotNull
  @Override
  public NTuple2<T2, T3> _23()  {
    return new NTuple2<>(elem2, elem3);
  }

  @NotNull
  @Override
  public NTuple2<T3, T2> _32()  {
    return new NTuple2<>(elem3, elem2);
  }

  @NotNull
  @Override
  public NTuple2<T1, T4> _14()  {
    return new NTuple2<>(elem1, elem4);
  }

  @NotNull
  @Override
  public NTuple2<T4, T1> _41()  {
    return new NTuple2<>(elem4, elem1);
  }

  @NotNull
  @Override
  public NTuple2<T2, T4> _24()  {
    return new NTuple2<>(elem2, elem4);
  }

  @NotNull
  @Override
  public NTuple2<T4, T2> _42()  {
    return new NTuple2<>(elem4, elem2);
  }

  @NotNull
  @Override
  public NTuple2<T3, T4> _34()  {
    return new NTuple2<>(elem3, elem4);
  }

  @NotNull
  @Override
  public NTuple2<T4, T3> _43()  {
    return new NTuple2<>(elem4, elem3);
  }

  @NotNull
  @Override
  public NTuple2<T1, T5> _15()  {
    return new NTuple2<>(elem1, elem5);
  }

  @NotNull
  @Override
  public NTuple2<T5, T1> _51()  {
    return new NTuple2<>(elem5, elem1);
  }

  @NotNull
  @Override
  public NTuple2<T2, T5> _25()  {
    return new NTuple2<>(elem2, elem5);
  }

  @NotNull
  @Override
  public NTuple2<T5, T2> _52()  {
    return new NTuple2<>(elem5, elem2);
  }

  @NotNull
  @Override
  public NTuple2<T3, T5> _35()  {
    return new NTuple2<>(elem3, elem5);
  }

  @NotNull
  @Override
  public NTuple2<T5, T3> _53()  {
    return new NTuple2<>(elem5, elem3);
  }

  @NotNull
  @Override
  public NTuple2<T4, T5> _45()  {
    return new NTuple2<>(elem4, elem5);
  }

  @NotNull
  @Override
  public NTuple2<T5, T4> _54()  {
    return new NTuple2<>(elem5, elem4);
  }

  @NotNull
  @Override
  public NTuple2<T1, T6> _16()  {
    return new NTuple2<>(elem1, elem6);
  }

  @NotNull
  @Override
  public NTuple2<T6, T1> _61()  {
    return new NTuple2<>(elem6, elem1);
  }

  @NotNull
  @Override
  public NTuple2<T2, T6> _26()  {
    return new NTuple2<>(elem2, elem6);
  }

  @NotNull
  @Override
  public NTuple2<T6, T2> _62()  {
    return new NTuple2<>(elem6, elem2);
  }

  @NotNull
  @Override
  public NTuple2<T3, T6> _36()  {
    return new NTuple2<>(elem3, elem6);
  }

  @NotNull
  @Override
  public NTuple2<T6, T3> _63()  {
    return new NTuple2<>(elem6, elem3);
  }

  @NotNull
  @Override
  public NTuple2<T4, T6> _46()  {
    return new NTuple2<>(elem4, elem6);
  }

  @NotNull
  @Override
  public NTuple2<T6, T4> _64()  {
    return new NTuple2<>(elem6, elem4);
  }

  @NotNull
  @Override
  public NTuple2<T5, T6> _56()  {
    return new NTuple2<>(elem5, elem6);
  }

  @NotNull
  @Override
  public NTuple2<T6, T5> _65()  {
    return new NTuple2<>(elem6, elem5);
  }

  @NotNull
  @Override
  public NTuple6<T1, T2, T3, T4, T5, T6> frozen()
  {
    return this;
  }

  /**
   * Create a 6-tuple from this one where the first element is exchanged. 
   * @param e1 new first element
   * @return tuple with exchanged element
   * @param <E1> type of exchanged first element
   */
  @NotNull
  public <E1> NTuple6<E1, T2, T3, T4, T5, T6> x1(@Nullable E1 e1)  {
    return new NTuple6<>(e1, elem2, elem3, elem4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the second element is exchanged. 
   * @param e2 new second element
   * @return tuple with exchanged element
   * @param <E2> type of exchanged second element
   */
  @NotNull
  public <E2> NTuple6<T1, E2, T3, T4, T5, T6> x2(@Nullable E2 e2)  {
    return new NTuple6<>(elem1, e2, elem3, elem4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the third element is exchanged. 
   * @param e3 new third element
   * @return tuple with exchanged element
   * @param <E3> type of exchanged third element
   */
  @NotNull
  public <E3> NTuple6<T1, T2, E3, T4, T5, T6> x3(@Nullable E3 e3)  {
    return new NTuple6<>(elem1, elem2, e3, elem4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the fourth element is exchanged. 
   * @param e4 new fourth element
   * @return tuple with exchanged element
   * @param <E4> type of exchanged fourth element
   */
  @NotNull
  public <E4> NTuple6<T1, T2, T3, E4, T5, T6> x4(@Nullable E4 e4)  {
    return new NTuple6<>(elem1, elem2, elem3, e4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the fifth element is exchanged. 
   * @param e5 new fifth element
   * @return tuple with exchanged element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E5> NTuple6<T1, T2, T3, T4, E5, T6> x5(@Nullable E5 e5)  {
    return new NTuple6<>(elem1, elem2, elem3, elem4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the sixth element is exchanged. 
   * @param e6 new sixth element
   * @return tuple with exchanged element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E6> NTuple6<T1, T2, T3, T4, T5, E6> x6(@Nullable E6 e6)  {
    return new NTuple6<>(elem1, elem2, elem3, elem4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first and second elements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   */
  @NotNull
  public <E1, E2> NTuple6<E1, E2, T3, T4, T5, T6> x12(@Nullable E1 e1, @Nullable E2 e2)  {
    return new NTuple6<>(e1, e2, elem3, elem4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first and third elements are exchanged. 
   * @param e1 new first element
   * @param e3 new third element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E3> type of exchanged third element
   */
  @NotNull
  public <E1, E3> NTuple6<E1, T2, E3, T4, T5, T6> x13(@Nullable E1 e1, @Nullable E3 e3)  {
    return new NTuple6<>(e1, elem2, e3, elem4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the second and third elements are exchanged. 
   * @param e2 new second element
   * @param e3 new third element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   */
  @NotNull
  public <E2, E3> NTuple6<T1, E2, E3, T4, T5, T6> x23(@Nullable E2 e2, @Nullable E3 e3)  {
    return new NTuple6<>(elem1, e2, e3, elem4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first and fourth elements are exchanged. 
   * @param e1 new first element
   * @param e4 new fourth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E4> type of exchanged fourth element
   */
  @NotNull
  public <E1, E4> NTuple6<E1, T2, T3, E4, T5, T6> x14(@Nullable E1 e1, @Nullable E4 e4)  {
    return new NTuple6<>(e1, elem2, elem3, e4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the second and fourth elements are exchanged. 
   * @param e2 new second element
   * @param e4 new fourth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E4> type of exchanged fourth element
   */
  @NotNull
  public <E2, E4> NTuple6<T1, E2, T3, E4, T5, T6> x24(@Nullable E2 e2, @Nullable E4 e4)  {
    return new NTuple6<>(elem1, e2, elem3, e4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the third and fourth elements are exchanged. 
   * @param e3 new third element
   * @param e4 new fourth element
   * @return tuple with exchanged elements
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   */
  @NotNull
  public <E3, E4> NTuple6<T1, T2, E3, E4, T5, T6> x34(@Nullable E3 e3, @Nullable E4 e4)  {
    return new NTuple6<>(elem1, elem2, e3, e4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first and fifth elements are exchanged. 
   * @param e1 new first element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E1, E5> NTuple6<E1, T2, T3, T4, E5, T6> x15(@Nullable E1 e1, @Nullable E5 e5)  {
    return new NTuple6<>(e1, elem2, elem3, elem4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the second and fifth elements are exchanged. 
   * @param e2 new second element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E2, E5> NTuple6<T1, E2, T3, T4, E5, T6> x25(@Nullable E2 e2, @Nullable E5 e5)  {
    return new NTuple6<>(elem1, e2, elem3, elem4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the third and fifth elements are exchanged. 
   * @param e3 new third element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E3> type of exchanged third element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E3, E5> NTuple6<T1, T2, E3, T4, E5, T6> x35(@Nullable E3 e3, @Nullable E5 e5)  {
    return new NTuple6<>(elem1, elem2, e3, elem4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the fourth and fifth elements are exchanged. 
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E4, E5> NTuple6<T1, T2, T3, E4, E5, T6> x45(@Nullable E4 e4, @Nullable E5 e5)  {
    return new NTuple6<>(elem1, elem2, elem3, e4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first and sixth elements are exchanged. 
   * @param e1 new first element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E6> NTuple6<E1, T2, T3, T4, T5, E6> x16(@Nullable E1 e1, @Nullable E6 e6)  {
    return new NTuple6<>(e1, elem2, elem3, elem4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the second and sixth elements are exchanged. 
   * @param e2 new second element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E2, E6> NTuple6<T1, E2, T3, T4, T5, E6> x26(@Nullable E2 e2, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, e2, elem3, elem4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the third and sixth elements are exchanged. 
   * @param e3 new third element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E3> type of exchanged third element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E3, E6> NTuple6<T1, T2, E3, T4, T5, E6> x36(@Nullable E3 e3, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, elem2, e3, elem4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the fourth and sixth elements are exchanged. 
   * @param e4 new fourth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E4> type of exchanged fourth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E4, E6> NTuple6<T1, T2, T3, E4, T5, E6> x46(@Nullable E4 e4, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, elem2, elem3, e4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the fifth and sixth elements are exchanged. 
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E5, E6> NTuple6<T1, T2, T3, T4, E5, E6> x56(@Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, elem2, elem3, elem4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, and thirdelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e3 new third element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   */
  @NotNull
  public <E1, E2, E3> NTuple6<E1, E2, E3, T4, T5, T6> x123(@Nullable E1 e1, @Nullable E2 e2, @Nullable E3 e3)  {
    return new NTuple6<>(e1, e2, e3, elem4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, and fourthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e4 new fourth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E4> type of exchanged fourth element
   */
  @NotNull
  public <E1, E2, E4> NTuple6<E1, E2, T3, E4, T5, T6> x124(@Nullable E1 e1, @Nullable E2 e2, @Nullable E4 e4)  {
    return new NTuple6<>(e1, e2, elem3, e4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first, third, and fourthelements are exchanged. 
   * @param e1 new first element
   * @param e3 new third element
   * @param e4 new fourth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   */
  @NotNull
  public <E1, E3, E4> NTuple6<E1, T2, E3, E4, T5, T6> x134(@Nullable E1 e1, @Nullable E3 e3, @Nullable E4 e4)  {
    return new NTuple6<>(e1, elem2, e3, e4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the second, third, and fourthelements are exchanged. 
   * @param e2 new second element
   * @param e3 new third element
   * @param e4 new fourth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   */
  @NotNull
  public <E2, E3, E4> NTuple6<T1, E2, E3, E4, T5, T6> x234(@Nullable E2 e2, @Nullable E3 e3, @Nullable E4 e4)  {
    return new NTuple6<>(elem1, e2, e3, e4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, and fifthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E1, E2, E5> NTuple6<E1, E2, T3, T4, E5, T6> x125(@Nullable E1 e1, @Nullable E2 e2, @Nullable E5 e5)  {
    return new NTuple6<>(e1, e2, elem3, elem4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first, third, and fifthelements are exchanged. 
   * @param e1 new first element
   * @param e3 new third element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E3> type of exchanged third element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E1, E3, E5> NTuple6<E1, T2, E3, T4, E5, T6> x135(@Nullable E1 e1, @Nullable E3 e3, @Nullable E5 e5)  {
    return new NTuple6<>(e1, elem2, e3, elem4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the second, third, and fifthelements are exchanged. 
   * @param e2 new second element
   * @param e3 new third element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E2, E3, E5> NTuple6<T1, E2, E3, T4, E5, T6> x235(@Nullable E2 e2, @Nullable E3 e3, @Nullable E5 e5)  {
    return new NTuple6<>(elem1, e2, e3, elem4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first, fourth, and fifthelements are exchanged. 
   * @param e1 new first element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E1, E4, E5> NTuple6<E1, T2, T3, E4, E5, T6> x145(@Nullable E1 e1, @Nullable E4 e4, @Nullable E5 e5)  {
    return new NTuple6<>(e1, elem2, elem3, e4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the second, fourth, and fifthelements are exchanged. 
   * @param e2 new second element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E2, E4, E5> NTuple6<T1, E2, T3, E4, E5, T6> x245(@Nullable E2 e2, @Nullable E4 e4, @Nullable E5 e5)  {
    return new NTuple6<>(elem1, e2, elem3, e4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the third, fourth, and fifthelements are exchanged. 
   * @param e3 new third element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E3, E4, E5> NTuple6<T1, T2, E3, E4, E5, T6> x345(@Nullable E3 e3, @Nullable E4 e4, @Nullable E5 e5)  {
    return new NTuple6<>(elem1, elem2, e3, e4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E2, E6> NTuple6<E1, E2, T3, T4, T5, E6> x126(@Nullable E1 e1, @Nullable E2 e2, @Nullable E6 e6)  {
    return new NTuple6<>(e1, e2, elem3, elem4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, third, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e3 new third element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E3> type of exchanged third element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E3, E6> NTuple6<E1, T2, E3, T4, T5, E6> x136(@Nullable E1 e1, @Nullable E3 e3, @Nullable E6 e6)  {
    return new NTuple6<>(e1, elem2, e3, elem4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the second, third, and sixthelements are exchanged. 
   * @param e2 new second element
   * @param e3 new third element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E2, E3, E6> NTuple6<T1, E2, E3, T4, T5, E6> x236(@Nullable E2 e2, @Nullable E3 e3, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, e2, e3, elem4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, fourth, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e4 new fourth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E4> type of exchanged fourth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E4, E6> NTuple6<E1, T2, T3, E4, T5, E6> x146(@Nullable E1 e1, @Nullable E4 e4, @Nullable E6 e6)  {
    return new NTuple6<>(e1, elem2, elem3, e4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the second, fourth, and sixthelements are exchanged. 
   * @param e2 new second element
   * @param e4 new fourth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E4> type of exchanged fourth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E2, E4, E6> NTuple6<T1, E2, T3, E4, T5, E6> x246(@Nullable E2 e2, @Nullable E4 e4, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, e2, elem3, e4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the third, fourth, and sixthelements are exchanged. 
   * @param e3 new third element
   * @param e4 new fourth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E3, E4, E6> NTuple6<T1, T2, E3, E4, T5, E6> x346(@Nullable E3 e3, @Nullable E4 e4, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, elem2, e3, e4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, fifth, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E5, E6> NTuple6<E1, T2, T3, T4, E5, E6> x156(@Nullable E1 e1, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(e1, elem2, elem3, elem4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the second, fifth, and sixthelements are exchanged. 
   * @param e2 new second element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E2, E5, E6> NTuple6<T1, E2, T3, T4, E5, E6> x256(@Nullable E2 e2, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, e2, elem3, elem4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the third, fifth, and sixthelements are exchanged. 
   * @param e3 new third element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E3> type of exchanged third element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E3, E5, E6> NTuple6<T1, T2, E3, T4, E5, E6> x356(@Nullable E3 e3, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, elem2, e3, elem4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the fourth, fifth, and sixthelements are exchanged. 
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E4, E5, E6> NTuple6<T1, T2, T3, E4, E5, E6> x456(@Nullable E4 e4, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, elem2, elem3, e4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, third, and fourthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e3 new third element
   * @param e4 new fourth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   */
  @NotNull
  public <E1, E2, E3, E4> NTuple6<E1, E2, E3, E4, T5, T6> x1234(@Nullable E1 e1, @Nullable E2 e2, @Nullable E3 e3, @Nullable E4 e4)  {
    return new NTuple6<>(e1, e2, e3, e4, elem5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, third, and fifthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e3 new third element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E1, E2, E3, E5> NTuple6<E1, E2, E3, T4, E5, T6> x1235(@Nullable E1 e1, @Nullable E2 e2, @Nullable E3 e3, @Nullable E5 e5)  {
    return new NTuple6<>(e1, e2, e3, elem4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, fourth, and fifthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E1, E2, E4, E5> NTuple6<E1, E2, T3, E4, E5, T6> x1245(@Nullable E1 e1, @Nullable E2 e2, @Nullable E4 e4, @Nullable E5 e5)  {
    return new NTuple6<>(e1, e2, elem3, e4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first, third, fourth, and fifthelements are exchanged. 
   * @param e1 new first element
   * @param e3 new third element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E1, E3, E4, E5> NTuple6<E1, T2, E3, E4, E5, T6> x1345(@Nullable E1 e1, @Nullable E3 e3, @Nullable E4 e4, @Nullable E5 e5)  {
    return new NTuple6<>(e1, elem2, e3, e4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the second, third, fourth, and fifthelements are exchanged. 
   * @param e2 new second element
   * @param e3 new third element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E2, E3, E4, E5> NTuple6<T1, E2, E3, E4, E5, T6> x2345(@Nullable E2 e2, @Nullable E3 e3, @Nullable E4 e4, @Nullable E5 e5)  {
    return new NTuple6<>(elem1, e2, e3, e4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, third, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e3 new third element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E2, E3, E6> NTuple6<E1, E2, E3, T4, T5, E6> x1236(@Nullable E1 e1, @Nullable E2 e2, @Nullable E3 e3, @Nullable E6 e6)  {
    return new NTuple6<>(e1, e2, e3, elem4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, fourth, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e4 new fourth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E4> type of exchanged fourth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E2, E4, E6> NTuple6<E1, E2, T3, E4, T5, E6> x1246(@Nullable E1 e1, @Nullable E2 e2, @Nullable E4 e4, @Nullable E6 e6)  {
    return new NTuple6<>(e1, e2, elem3, e4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, third, fourth, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e3 new third element
   * @param e4 new fourth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E3, E4, E6> NTuple6<E1, T2, E3, E4, T5, E6> x1346(@Nullable E1 e1, @Nullable E3 e3, @Nullable E4 e4, @Nullable E6 e6)  {
    return new NTuple6<>(e1, elem2, e3, e4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the second, third, fourth, and sixthelements are exchanged. 
   * @param e2 new second element
   * @param e3 new third element
   * @param e4 new fourth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E2, E3, E4, E6> NTuple6<T1, E2, E3, E4, T5, E6> x2346(@Nullable E2 e2, @Nullable E3 e3, @Nullable E4 e4, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, e2, e3, e4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, fifth, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E2, E5, E6> NTuple6<E1, E2, T3, T4, E5, E6> x1256(@Nullable E1 e1, @Nullable E2 e2, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(e1, e2, elem3, elem4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, third, fifth, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e3 new third element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E3> type of exchanged third element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E3, E5, E6> NTuple6<E1, T2, E3, T4, E5, E6> x1356(@Nullable E1 e1, @Nullable E3 e3, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(e1, elem2, e3, elem4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the second, third, fifth, and sixthelements are exchanged. 
   * @param e2 new second element
   * @param e3 new third element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E2, E3, E5, E6> NTuple6<T1, E2, E3, T4, E5, E6> x2356(@Nullable E2 e2, @Nullable E3 e3, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, e2, e3, elem4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, fourth, fifth, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E4, E5, E6> NTuple6<E1, T2, T3, E4, E5, E6> x1456(@Nullable E1 e1, @Nullable E4 e4, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(e1, elem2, elem3, e4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the second, fourth, fifth, and sixthelements are exchanged. 
   * @param e2 new second element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E2, E4, E5, E6> NTuple6<T1, E2, T3, E4, E5, E6> x2456(@Nullable E2 e2, @Nullable E4 e4, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, e2, elem3, e4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the third, fourth, fifth, and sixthelements are exchanged. 
   * @param e3 new third element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E3, E4, E5, E6> NTuple6<T1, T2, E3, E4, E5, E6> x3456(@Nullable E3 e3, @Nullable E4 e4, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, elem2, e3, e4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, third, fourth, and fifthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e3 new third element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   */
  @NotNull
  public <E1, E2, E3, E4, E5> NTuple6<E1, E2, E3, E4, E5, T6> x12345(@Nullable E1 e1, @Nullable E2 e2, @Nullable E3 e3, @Nullable E4 e4, @Nullable E5 e5)  {
    return new NTuple6<>(e1, e2, e3, e4, e5, elem6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, third, fourth, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e3 new third element
   * @param e4 new fourth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E2, E3, E4, E6> NTuple6<E1, E2, E3, E4, T5, E6> x12346(@Nullable E1 e1, @Nullable E2 e2, @Nullable E3 e3, @Nullable E4 e4, @Nullable E6 e6)  {
    return new NTuple6<>(e1, e2, e3, e4, elem5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, third, fifth, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e3 new third element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E2, E3, E5, E6> NTuple6<E1, E2, E3, T4, E5, E6> x12356(@Nullable E1 e1, @Nullable E2 e2, @Nullable E3 e3, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(e1, e2, e3, elem4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, second, fourth, fifth, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e2 new second element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E2> type of exchanged second element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E2, E4, E5, E6> NTuple6<E1, E2, T3, E4, E5, E6> x12456(@Nullable E1 e1, @Nullable E2 e2, @Nullable E4 e4, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(e1, e2, elem3, e4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the first, third, fourth, fifth, and sixthelements are exchanged. 
   * @param e1 new first element
   * @param e3 new third element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E1> type of exchanged first element
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E1, E3, E4, E5, E6> NTuple6<E1, T2, E3, E4, E5, E6> x13456(@Nullable E1 e1, @Nullable E3 e3, @Nullable E4 e4, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(e1, elem2, e3, e4, e5, e6);
  }

  /**
   * Create a 6-tuple from this one where the second, third, fourth, fifth, and sixthelements are exchanged. 
   * @param e2 new second element
   * @param e3 new third element
   * @param e4 new fourth element
   * @param e5 new fifth element
   * @param e6 new sixth element
   * @return tuple with exchanged elements
   * @param <E2> type of exchanged second element
   * @param <E3> type of exchanged third element
   * @param <E4> type of exchanged fourth element
   * @param <E5> type of exchanged fifth element
   * @param <E6> type of exchanged sixth element
   */
  @NotNull
  public <E2, E3, E4, E5, E6> NTuple6<T1, E2, E3, E4, E5, E6> x23456(@Nullable E2 e2, @Nullable E3 e3, @Nullable E4 e4, @Nullable E5 e5, @Nullable E6 e6)  {
    return new NTuple6<>(elem1, e2, e3, e4, e5, e6);
  }

  /**
   * Create a 7-tuple from this 6-tuple by inserting a new first element.
   * @param e1 new element
   * @param <T> new element type
   * @return 7-tuple
   */
  @NotNull
  public <T> NTuple7<T, T1, T2, T3, T4, T5, T6> i1(@Nullable T e1)
  {
    return new NTuple7<>(e1, elem1, elem2, elem3, elem4, elem5, elem6);
  }

  /**
   * Create a 7-tuple from this 6-tuple by inserting a new second element.
   * @param e2 new element
   * @param <T> new element type
   * @return 7-tuple
   */
  @NotNull
  public <T> NTuple7<T1, T, T2, T3, T4, T5, T6> i2(@Nullable T e2)
  {
    return new NTuple7<>(elem1, e2, elem2, elem3, elem4, elem5, elem6);
  }

  /**
   * Create a 7-tuple from this 6-tuple by inserting a new third element.
   * @param e3 new element
   * @param <T> new element type
   * @return 7-tuple
   */
  @NotNull
  public <T> NTuple7<T1, T2, T, T3, T4, T5, T6> i3(@Nullable T e3)
  {
    return new NTuple7<>(elem1, elem2, e3, elem3, elem4, elem5, elem6);
  }

  /**
   * Create a 7-tuple from this 6-tuple by inserting a new fourth element.
   * @param e4 new element
   * @param <T> new element type
   * @return 7-tuple
   */
  @NotNull
  public <T> NTuple7<T1, T2, T3, T, T4, T5, T6> i4(@Nullable T e4)
  {
    return new NTuple7<>(elem1, elem2, elem3, e4, elem4, elem5, elem6);
  }

  /**
   * Create a 7-tuple from this 6-tuple by inserting a new fifth element.
   * @param e5 new element
   * @param <T> new element type
   * @return 7-tuple
   */
  @NotNull
  public <T> NTuple7<T1, T2, T3, T4, T, T5, T6> i5(@Nullable T e5)
  {
    return new NTuple7<>(elem1, elem2, elem3, elem4, e5, elem5, elem6);
  }

  /**
   * Create a 7-tuple from this 6-tuple by inserting a new sixth element.
   * @param e6 new element
   * @param <T> new element type
   * @return 7-tuple
   */
  @NotNull
  public <T> NTuple7<T1, T2, T3, T4, T5, T, T6> i6(@Nullable T e6)
  {
    return new NTuple7<>(elem1, elem2, elem3, elem4, elem5, e6, elem6);
  }

  /**
   * Create a 7-tuple from this 6-tuple by adding a new seventh element.
   * @param e7 new element
   * @param <T> new element type
   * @return 7-tuple
   */
  @NotNull
  public <T> NTuple7<T1, T2, T3, T4, T5, T6, T> i7(@Nullable T e7)
  {
    return new NTuple7<>(elem1, elem2, elem3, elem4, elem5, elem6, e7);
  }

  @NotNull
  @Override
  public String toString()
  {
    return ITuple6.toString("NTuple6", this);
  }

}
