// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021 - 2022  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.tuple;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.*;

import java.util.Objects;
import java.util.function.*;

/**
 * General interface for immutable 2-tuples.
 * <p>
 * This is sometimes useful to access either {@link Tuple2} or {@link NTuple2} in a general way,
 * although in most cases the concrete implementations are usually preferable.
 * <p>
 * If you want to iterate over a tuple or access its elements by index
 * (note that both is only possible for a common super type of the tuple's types)
 * {@link de.caff.generics.Indexable#viewTuple(ITuple2)} will come to help.
 * <p>
 * Note that this class is automatically created by {@code tools.TupleClassCreator} (not yet public).
 *
 * @param <T1> type of first element
 * @param <T2> type of second element
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 17, 2021
 * @see Tuple2
 * @see NTuple2
 */
public interface ITuple2<T1, T2>
{
  /**
   * Get the first element.
   * @return first element
   */
  T1 _1();

  /**
   * Get the second element.
   * @return second element
   */
  T2 _2();

  /**
   * Create a new 2-tuple from this one where the elements are ordered: 2nd, 1st.
   * @return reordered tuple
   */
  @NotNull
  default ITuple2<T2, T1> _21()  {
    return new ITuple2.Base<T2, T1>() {
      @Override
      public T2 _1()
      {
        return ITuple2.this._2();
      }

      @Override
      public T1 _2()
      {
        return ITuple2.this._1();
      }
    };
  }

  /**
   * Convert this 2-tuple into a {@link ITuple2.Base base tuple} which provides useful implementations for standard methods.
   * @return base view of this tuple
   */
  @NotNull
  default ITuple2.Base<T1, T2> asBase()
  {
    return new Base<T1, T2>()
    {
      @Override
      public T1 _1()
      {
        return ITuple2.this._1();
      }

      @Override
      public T2 _2()
      {
        return ITuple2.this._2();
      }
    };
  }

  /**
   * Convert this into an object of a concrete tuple implementation.
   * This is useful because implementations may just be a view of something else.
   * This method makes sure to decouple the tuple from the viewed object.
   * The resulting tuple may contain {@code null} elements,
   * see {@link #frozenNotNull()} for a method which returns tuples
   * which don't have nullable elements.
   * @return standalone tuple object
   */
  @NotNull
  default NTuple2<T1, T2> frozen()
  {
    return new NTuple2<>(_1(), _2());
  }

  /**
   * Convert this into an object of a concrete tuple implementation.
   * This is useful because implementations may just be a view of something else.
   * This method makes sure to decouple the tuple from the viewed object.
   * The resulting tuple is guaranteed not to have {@code null}
   * elements, but if this interface has {@code null} elements this method
   * will throw a {@link NullPointerException}.
   *<p>
   * See {@link #frozen()} for a method which will not throw exceptions.
   * @return standalone tuple object
   * @throws NullPointerException if any element of this tuple is {@code null}
   */
  @NotNull
  default Tuple2<T1, T2> frozenNotNull()
  {
    return new Tuple2<>(Objects.requireNonNull(_1(), "_1()"),
                        Objects.requireNonNull(_2(), "_2()"));
  }

  /**
   * Invoke a 2-argument function on this tuple unpacked.
   * @param function function called with the unpacked elements of this tuple
   * @param <T> function return type
   * @return result of calling the function
   */
  default <T> T invoke(@NotNull BiFunction<? super T1, ? super T2, T> function)
  {
    return function.apply(_1(), _2());
  }

  /**
   * Send this tuple unpacked to a 2-argument procedure.
   * @param procedure procedure called with the unpacked elements of this tuple
   */
  default void sendTo(@NotNull BiConsumer<? super T1, ? super T2> procedure)
  {
    procedure.accept(_1(), _2());
  }

  /**
   * Test a 2-argument predicate with this tuple unpacked.
   * @param predicate predicate called with the unpacked elements of this tuple
   * @return predicate's result
   */
  default boolean testBy(@NotNull BiPredicate<? super T1, ? super T2> predicate)
  {
    return predicate.test(_1(), _2());
  }

  /**
   * Create a new 2-tuple on the fly.
   * This is especially useful for temporary usage, you might want to create a {@link Tuple2} 
   * instead as that gives guarantees of the non-nullness of its elements, or a {@link NTuple2}
   * as that at least is serializable (as is {@code Tuple2}). 
   * @param elem1 first element of returned tuple
   * @param elem2 second element of returned tuple
   * @param <E1> type of first element
   * @param <E2> type of second element
   * @return 2-tuple view of the given elements
   */
  @NotNull
  static <E1, E2> ITuple2<E1, E2> view(E1 elem1, E2 elem2)
  {
    return new ITuple2.Base<E1, E2>() {
      @Override
      public E1 _1()
      {
        return elem1;
      }

      @Override
      public E2 _2()
      {
        return elem2;
      }
    };
  }

  /**
   * Basic implementation of equals for 2-tuples.
   * This can be used in implementations to implement {@link Object#equals(Object)}.
   * Don't forget hashing: {@link #hash(ITuple2)}
   * @param tuple tuple to compare
   * @param o     other object to compare
   * @return {@code false} if {@code o} is {@code null}, not a {@code ITuple2}, or its members differ<br>
   *         {@code true} otherwise
   */
  static boolean equals(@NotNull ITuple2<?, ?> tuple, @Nullable Object o)
  {
    if (tuple == o) {
      return true;
    }
    if (!(o instanceof ITuple2)) {
      // null will come here, too
      return false;
    }
    final ITuple2<?, ?> other = (ITuple2<?, ?>)o;
    return Objects.equals(tuple._1(), other._1())  &&  Objects.equals(tuple._2(), other._2());
  }

  /**
   * Basic implementation for calculating a hash code of a 2-tuple.
   * This can be used in implementations to implement {@link Object#hashCode()}.
   * Don't forget equals: {@link #equals(ITuple2, Object)}.
   * @param tuple tuple to hash
   * @return hash code
   */
  static int hash(@NotNull ITuple2<?, ?> tuple)
  {
    return Objects.hash(tuple._1(), tuple._2());
  }

  /**
   * Basic implementation for creating a string from a 2-tuple.
   * This one uses {@code "ITuple2"} as prefix.
   * This can be used im implementations to implement {@link Object#toString()}.
   * @param tuple tuple to convert
   * @return text form
   * @see #toString(String, ITuple2)
   */
  @NotNull
  static String toString(@NotNull ITuple2<?, ?> tuple)
  {
    return toString("ITuple2", tuple);
  }

  /**
   * Basic implementation for creating a string from a 2-tuple.
   * This can be used im implementations to implement {@link Object#toString()}.
   * @param prefix prefix for the output
   * @param tuple tuple to convert
   * @return text form
   * @see #toString(ITuple2)
   */
  @NotNull
  static String toString(@NotNull String prefix,
                         @NotNull ITuple2<?, ?> tuple)
  {
    return prefix + '(' + tuple._1() +  ',' + tuple._2() +  ')';
  }

  /**
   * Combine a 1-argument function returning a 2-tuple with a 2-parameter function returning another result value
   * into one 1-argument function.
   * @param fn1  1-argument function returning a 2-tuple
   * @param fn2  2-argument function which can use the expanded tuple elements as arguments and returns the result value 
   * @return 1-argument function which returns the result value 
   * @param <S1> first argument of incoming function {@code fn1} and returned function
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   * @param <R> result type
   */
  @NotNull
  static <S1, E1, E2, R>
  Function<S1, R> concat1(@NotNull Function<S1, ? extends ITuple2<? extends E1, ? extends E2>> fn1,
                          @NotNull BiFunction<? super E1, ? super E2, R> fn2)
  {
    return fn1.andThen(tuple -> tuple.invoke(fn2));
  }

  /**
   * Combine a 1-argument function returning a 2-tuple with a 2-parameter consumer/procedure into a 1-argument consumer/procedure.
   * @param fn1    1-argument function returning a 2-tuple
   * @param proc2  2-argument consumer/procedure which can use the expanded tuple elements as arguments 
   * @return 1-argument consumer/procedure which returns the result value 
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, E1, E2>
  Consumer<S1> concatProc1(@NotNull Function<S1, ? extends ITuple2<? extends E1, ? extends E2>> fn1,
                           @NotNull BiConsumer<? super E1, ? super E2> proc2)
  {
    return (v1) -> fn1.apply(v1).sendTo(proc2);
  }

  /**
   * Combine a 1-argument function returning a 2-tuple with a 2-parameter predicate into a 1-argument predicate.
   * @param fn1    1-argument function returning a 2-tuple
   * @param pred2  2-argument predicate which can use the expanded tuple elements as arguments 
   * @return 1-argument predicate which returns the test result of the tuple returned by {@code fn1}
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, E1, E2>
  Predicate<S1> concatPred1(@NotNull Function<S1, ? extends ITuple2<? extends E1, ? extends E2>> fn1,
                            @NotNull BiPredicate<? super E1, ? super E2> pred2)
  {
    return (v1) -> fn1.apply(v1).testBy(pred2);
  }

  /**
   * Combine a 2-argument function returning a 2-tuple with a 2-parameter function returning another result value
   * into one 2-argument function.
   * @param fn1  2-argument function returning a 2-tuple
   * @param fn2  2-argument function which can use the expanded tuple elements as arguments and returns the result value 
   * @return 2-argument function which returns the result value 
   * @param <S1> first argument of incoming function {@code fn1} and returned function
   * @param <S2> second argument of incoming function {@code fn1} and returned function
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   * @param <R> result type
   */
  @NotNull
  static <S1, S2, E1, E2, R>
  BiFunction<S1, S2, R> concat2(@NotNull BiFunction<S1, S2, ? extends ITuple2<? extends E1, ? extends E2>> fn1,
                                @NotNull BiFunction<? super E1, ? super E2, R> fn2)
  {
    return fn1.andThen(tuple -> tuple.invoke(fn2));
  }

  /**
   * Combine a 2-argument function returning a 2-tuple with a 2-parameter consumer/procedure into a 2-argument consumer/procedure.
   * @param fn1    2-argument function returning a 2-tuple
   * @param proc2  2-argument consumer/procedure which can use the expanded tuple elements as arguments 
   * @return 2-argument consumer/procedure which returns the result value 
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, E1, E2>
  BiConsumer<S1, S2> concatProc2(@NotNull BiFunction<S1, S2, ? extends ITuple2<? extends E1, ? extends E2>> fn1,
                                 @NotNull BiConsumer<? super E1, ? super E2> proc2)
  {
    return (v1, v2) -> fn1.apply(v1, v2).sendTo(proc2);
  }

  /**
   * Combine a 2-argument function returning a 2-tuple with a 2-parameter predicate into a 2-argument predicate.
   * @param fn1    2-argument function returning a 2-tuple
   * @param pred2  2-argument predicate which can use the expanded tuple elements as arguments 
   * @return 2-argument predicate which returns the test result of the tuple returned by {@code fn1}
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, E1, E2>
  BiPredicate<S1, S2> concatPred2(@NotNull BiFunction<S1, S2, ? extends ITuple2<? extends E1, ? extends E2>> fn1,
                                  @NotNull BiPredicate<? super E1, ? super E2> pred2)
  {
    return (v1, v2) -> fn1.apply(v1, v2).testBy(pred2);
  }

  /**
   * Combine a 3-argument function returning a 2-tuple with a 2-parameter function returning another result value
   * into one 3-argument function.
   * @param fn1  3-argument function returning a 2-tuple
   * @param fn2  2-argument function which can use the expanded tuple elements as arguments and returns the result value 
   * @return 3-argument function which returns the result value 
   * @param <S1> first argument of incoming function {@code fn1} and returned function
   * @param <S2> second argument of incoming function {@code fn1} and returned function
   * @param <S3> third argument of incoming function {@code fn1} and returned function
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   * @param <R> result type
   */
  @NotNull
  static <S1, S2, S3, E1, E2, R>
  Function3<R, S1, S2, S3> concat3(@NotNull Function3<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3> fn1,
                                   @NotNull BiFunction<? super E1, ? super E2, R> fn2)
  {
    return fn1.andThen(tuple -> tuple.invoke(fn2));
  }

  /**
   * Combine a 3-argument function returning a 2-tuple with a 2-parameter consumer/procedure into a 3-argument consumer/procedure.
   * @param fn1    3-argument function returning a 2-tuple
   * @param proc2  2-argument consumer/procedure which can use the expanded tuple elements as arguments 
   * @return 3-argument consumer/procedure which returns the result value 
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <S3> third argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, S3, E1, E2>
  Procedure3<S1, S2, S3> concatProc3(@NotNull Function3<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3> fn1,
                                     @NotNull BiConsumer<? super E1, ? super E2> proc2)
  {
    return (v1, v2, v3) -> fn1.apply(v1, v2, v3).sendTo(proc2);
  }

  /**
   * Combine a 3-argument function returning a 2-tuple with a 2-parameter predicate into a 3-argument predicate.
   * @param fn1    3-argument function returning a 2-tuple
   * @param pred2  2-argument predicate which can use the expanded tuple elements as arguments 
   * @return 3-argument predicate which returns the test result of the tuple returned by {@code fn1}
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <S3> third argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, S3, E1, E2>
  Predicate3<S1, S2, S3> concatPred3(@NotNull Function3<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3> fn1,
                                     @NotNull BiPredicate<? super E1, ? super E2> pred2)
  {
    return (v1, v2, v3) -> fn1.apply(v1, v2, v3).testBy(pred2);
  }

  /**
   * Combine a 4-argument function returning a 2-tuple with a 2-parameter function returning another result value
   * into one 4-argument function.
   * @param fn1  4-argument function returning a 2-tuple
   * @param fn2  2-argument function which can use the expanded tuple elements as arguments and returns the result value 
   * @return 4-argument function which returns the result value 
   * @param <S1> first argument of incoming function {@code fn1} and returned function
   * @param <S2> second argument of incoming function {@code fn1} and returned function
   * @param <S3> third argument of incoming function {@code fn1} and returned function
   * @param <S4> fourth argument of incoming function {@code fn1} and returned function
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   * @param <R> result type
   */
  @NotNull
  static <S1, S2, S3, S4, E1, E2, R>
  Function4<R, S1, S2, S3, S4> concat4(@NotNull Function4<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4> fn1,
                                       @NotNull BiFunction<? super E1, ? super E2, R> fn2)
  {
    return fn1.andThen(tuple -> tuple.invoke(fn2));
  }

  /**
   * Combine a 4-argument function returning a 2-tuple with a 2-parameter consumer/procedure into a 4-argument consumer/procedure.
   * @param fn1    4-argument function returning a 2-tuple
   * @param proc2  2-argument consumer/procedure which can use the expanded tuple elements as arguments 
   * @return 4-argument consumer/procedure which returns the result value 
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <S3> third argument of function and returned consumer/procedure
   * @param <S4> fourth argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, S3, S4, E1, E2>
  Procedure4<S1, S2, S3, S4> concatProc4(@NotNull Function4<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4> fn1,
                                         @NotNull BiConsumer<? super E1, ? super E2> proc2)
  {
    return (v1, v2, v3, v4) -> fn1.apply(v1, v2, v3, v4).sendTo(proc2);
  }

  /**
   * Combine a 4-argument function returning a 2-tuple with a 2-parameter predicate into a 4-argument predicate.
   * @param fn1    4-argument function returning a 2-tuple
   * @param pred2  2-argument predicate which can use the expanded tuple elements as arguments 
   * @return 4-argument predicate which returns the test result of the tuple returned by {@code fn1}
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <S3> third argument of function and returned consumer/procedure
   * @param <S4> fourth argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, S3, S4, E1, E2>
  Predicate4<S1, S2, S3, S4> concatPred4(@NotNull Function4<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4> fn1,
                                         @NotNull BiPredicate<? super E1, ? super E2> pred2)
  {
    return (v1, v2, v3, v4) -> fn1.apply(v1, v2, v3, v4).testBy(pred2);
  }

  /**
   * Combine a 5-argument function returning a 2-tuple with a 2-parameter function returning another result value
   * into one 5-argument function.
   * @param fn1  5-argument function returning a 2-tuple
   * @param fn2  2-argument function which can use the expanded tuple elements as arguments and returns the result value 
   * @return 5-argument function which returns the result value 
   * @param <S1> first argument of incoming function {@code fn1} and returned function
   * @param <S2> second argument of incoming function {@code fn1} and returned function
   * @param <S3> third argument of incoming function {@code fn1} and returned function
   * @param <S4> fourth argument of incoming function {@code fn1} and returned function
   * @param <S5> fifth argument of incoming function {@code fn1} and returned function
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   * @param <R> result type
   */
  @NotNull
  static <S1, S2, S3, S4, S5, E1, E2, R>
  Function5<R, S1, S2, S3, S4, S5> concat5(@NotNull Function5<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4, S5> fn1,
                                           @NotNull BiFunction<? super E1, ? super E2, R> fn2)
  {
    return fn1.andThen(tuple -> tuple.invoke(fn2));
  }

  /**
   * Combine a 5-argument function returning a 2-tuple with a 2-parameter consumer/procedure into a 5-argument consumer/procedure.
   * @param fn1    5-argument function returning a 2-tuple
   * @param proc2  2-argument consumer/procedure which can use the expanded tuple elements as arguments 
   * @return 5-argument consumer/procedure which returns the result value 
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <S3> third argument of function and returned consumer/procedure
   * @param <S4> fourth argument of function and returned consumer/procedure
   * @param <S5> fifth argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, S3, S4, S5, E1, E2>
  Procedure5<S1, S2, S3, S4, S5> concatProc5(@NotNull Function5<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4, S5> fn1,
                                             @NotNull BiConsumer<? super E1, ? super E2> proc2)
  {
    return (v1, v2, v3, v4, v5) -> fn1.apply(v1, v2, v3, v4, v5).sendTo(proc2);
  }

  /**
   * Combine a 5-argument function returning a 2-tuple with a 2-parameter predicate into a 5-argument predicate.
   * @param fn1    5-argument function returning a 2-tuple
   * @param pred2  2-argument predicate which can use the expanded tuple elements as arguments 
   * @return 5-argument predicate which returns the test result of the tuple returned by {@code fn1}
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <S3> third argument of function and returned consumer/procedure
   * @param <S4> fourth argument of function and returned consumer/procedure
   * @param <S5> fifth argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, S3, S4, S5, E1, E2>
  Predicate5<S1, S2, S3, S4, S5> concatPred5(@NotNull Function5<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4, S5> fn1,
                                             @NotNull BiPredicate<? super E1, ? super E2> pred2)
  {
    return (v1, v2, v3, v4, v5) -> fn1.apply(v1, v2, v3, v4, v5).testBy(pred2);
  }

  /**
   * Combine a 6-argument function returning a 2-tuple with a 2-parameter function returning another result value
   * into one 6-argument function.
   * @param fn1  6-argument function returning a 2-tuple
   * @param fn2  2-argument function which can use the expanded tuple elements as arguments and returns the result value 
   * @return 6-argument function which returns the result value 
   * @param <S1> first argument of incoming function {@code fn1} and returned function
   * @param <S2> second argument of incoming function {@code fn1} and returned function
   * @param <S3> third argument of incoming function {@code fn1} and returned function
   * @param <S4> fourth argument of incoming function {@code fn1} and returned function
   * @param <S5> fifth argument of incoming function {@code fn1} and returned function
   * @param <S6> sixth argument of incoming function {@code fn1} and returned function
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   * @param <R> result type
   */
  @NotNull
  static <S1, S2, S3, S4, S5, S6, E1, E2, R>
  Function6<R, S1, S2, S3, S4, S5, S6> concat6(@NotNull Function6<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4, S5, S6> fn1,
                                               @NotNull BiFunction<? super E1, ? super E2, R> fn2)
  {
    return fn1.andThen(tuple -> tuple.invoke(fn2));
  }

  /**
   * Combine a 6-argument function returning a 2-tuple with a 2-parameter consumer/procedure into a 6-argument consumer/procedure.
   * @param fn1    6-argument function returning a 2-tuple
   * @param proc2  2-argument consumer/procedure which can use the expanded tuple elements as arguments 
   * @return 6-argument consumer/procedure which returns the result value 
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <S3> third argument of function and returned consumer/procedure
   * @param <S4> fourth argument of function and returned consumer/procedure
   * @param <S5> fifth argument of function and returned consumer/procedure
   * @param <S6> sixth argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, S3, S4, S5, S6, E1, E2>
  Procedure6<S1, S2, S3, S4, S5, S6> concatProc6(@NotNull Function6<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4, S5, S6> fn1,
                                                 @NotNull BiConsumer<? super E1, ? super E2> proc2)
  {
    return (v1, v2, v3, v4, v5, v6) -> fn1.apply(v1, v2, v3, v4, v5, v6).sendTo(proc2);
  }

  /**
   * Combine a 6-argument function returning a 2-tuple with a 2-parameter predicate into a 6-argument predicate.
   * @param fn1    6-argument function returning a 2-tuple
   * @param pred2  2-argument predicate which can use the expanded tuple elements as arguments 
   * @return 6-argument predicate which returns the test result of the tuple returned by {@code fn1}
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <S3> third argument of function and returned consumer/procedure
   * @param <S4> fourth argument of function and returned consumer/procedure
   * @param <S5> fifth argument of function and returned consumer/procedure
   * @param <S6> sixth argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, S3, S4, S5, S6, E1, E2>
  Predicate6<S1, S2, S3, S4, S5, S6> concatPred6(@NotNull Function6<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4, S5, S6> fn1,
                                                 @NotNull BiPredicate<? super E1, ? super E2> pred2)
  {
    return (v1, v2, v3, v4, v5, v6) -> fn1.apply(v1, v2, v3, v4, v5, v6).testBy(pred2);
  }

  /**
   * Combine a 7-argument function returning a 2-tuple with a 2-parameter function returning another result value
   * into one 7-argument function.
   * @param fn1  7-argument function returning a 2-tuple
   * @param fn2  2-argument function which can use the expanded tuple elements as arguments and returns the result value 
   * @return 7-argument function which returns the result value 
   * @param <S1> first argument of incoming function {@code fn1} and returned function
   * @param <S2> second argument of incoming function {@code fn1} and returned function
   * @param <S3> third argument of incoming function {@code fn1} and returned function
   * @param <S4> fourth argument of incoming function {@code fn1} and returned function
   * @param <S5> fifth argument of incoming function {@code fn1} and returned function
   * @param <S6> sixth argument of incoming function {@code fn1} and returned function
   * @param <S7> seventh argument of incoming function {@code fn1} and returned function
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   * @param <R> result type
   */
  @NotNull
  static <S1, S2, S3, S4, S5, S6, S7, E1, E2, R>
  Function7<R, S1, S2, S3, S4, S5, S6, S7> concat7(@NotNull Function7<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4, S5, S6, S7> fn1,
                                                   @NotNull BiFunction<? super E1, ? super E2, R> fn2)
  {
    return fn1.andThen(tuple -> tuple.invoke(fn2));
  }

  /**
   * Combine a 7-argument function returning a 2-tuple with a 2-parameter consumer/procedure into a 7-argument consumer/procedure.
   * @param fn1    7-argument function returning a 2-tuple
   * @param proc2  2-argument consumer/procedure which can use the expanded tuple elements as arguments 
   * @return 7-argument consumer/procedure which returns the result value 
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <S3> third argument of function and returned consumer/procedure
   * @param <S4> fourth argument of function and returned consumer/procedure
   * @param <S5> fifth argument of function and returned consumer/procedure
   * @param <S6> sixth argument of function and returned consumer/procedure
   * @param <S7> seventh argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, S3, S4, S5, S6, S7, E1, E2>
  Procedure7<S1, S2, S3, S4, S5, S6, S7> concatProc7(@NotNull Function7<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4, S5, S6, S7> fn1,
                                                     @NotNull BiConsumer<? super E1, ? super E2> proc2)
  {
    return (v1, v2, v3, v4, v5, v6, v7) -> fn1.apply(v1, v2, v3, v4, v5, v6, v7).sendTo(proc2);
  }

  /**
   * Combine a 7-argument function returning a 2-tuple with a 2-parameter predicate into a 7-argument predicate.
   * @param fn1    7-argument function returning a 2-tuple
   * @param pred2  2-argument predicate which can use the expanded tuple elements as arguments 
   * @return 7-argument predicate which returns the test result of the tuple returned by {@code fn1}
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <S3> third argument of function and returned consumer/procedure
   * @param <S4> fourth argument of function and returned consumer/procedure
   * @param <S5> fifth argument of function and returned consumer/procedure
   * @param <S6> sixth argument of function and returned consumer/procedure
   * @param <S7> seventh argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, S3, S4, S5, S6, S7, E1, E2>
  Predicate7<S1, S2, S3, S4, S5, S6, S7> concatPred7(@NotNull Function7<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4, S5, S6, S7> fn1,
                                                     @NotNull BiPredicate<? super E1, ? super E2> pred2)
  {
    return (v1, v2, v3, v4, v5, v6, v7) -> fn1.apply(v1, v2, v3, v4, v5, v6, v7).testBy(pred2);
  }

  /**
   * Combine a 8-argument function returning a 2-tuple with a 2-parameter function returning another result value
   * into one 8-argument function.
   * @param fn1  8-argument function returning a 2-tuple
   * @param fn2  2-argument function which can use the expanded tuple elements as arguments and returns the result value 
   * @return 8-argument function which returns the result value 
   * @param <S1> first argument of incoming function {@code fn1} and returned function
   * @param <S2> second argument of incoming function {@code fn1} and returned function
   * @param <S3> third argument of incoming function {@code fn1} and returned function
   * @param <S4> fourth argument of incoming function {@code fn1} and returned function
   * @param <S5> fifth argument of incoming function {@code fn1} and returned function
   * @param <S6> sixth argument of incoming function {@code fn1} and returned function
   * @param <S7> seventh argument of incoming function {@code fn1} and returned function
   * @param <S8> eighth argument of incoming function {@code fn1} and returned function
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   * @param <R> result type
   */
  @NotNull
  static <S1, S2, S3, S4, S5, S6, S7, S8, E1, E2, R>
  Function8<R, S1, S2, S3, S4, S5, S6, S7, S8> concat8(@NotNull Function8<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4, S5, S6, S7, S8> fn1,
                                                       @NotNull BiFunction<? super E1, ? super E2, R> fn2)
  {
    return fn1.andThen(tuple -> tuple.invoke(fn2));
  }

  /**
   * Combine a 8-argument function returning a 2-tuple with a 2-parameter consumer/procedure into a 8-argument consumer/procedure.
   * @param fn1    8-argument function returning a 2-tuple
   * @param proc2  2-argument consumer/procedure which can use the expanded tuple elements as arguments 
   * @return 8-argument consumer/procedure which returns the result value 
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <S3> third argument of function and returned consumer/procedure
   * @param <S4> fourth argument of function and returned consumer/procedure
   * @param <S5> fifth argument of function and returned consumer/procedure
   * @param <S6> sixth argument of function and returned consumer/procedure
   * @param <S7> seventh argument of function and returned consumer/procedure
   * @param <S8> eighth argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, S3, S4, S5, S6, S7, S8, E1, E2>
  Procedure8<S1, S2, S3, S4, S5, S6, S7, S8> concatProc8(@NotNull Function8<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4, S5, S6, S7, S8> fn1,
                                                         @NotNull BiConsumer<? super E1, ? super E2> proc2)
  {
    return (v1, v2, v3, v4, v5, v6, v7, v8) -> fn1.apply(v1, v2, v3, v4, v5, v6, v7, v8).sendTo(proc2);
  }

  /**
   * Combine a 8-argument function returning a 2-tuple with a 2-parameter predicate into a 8-argument predicate.
   * @param fn1    8-argument function returning a 2-tuple
   * @param pred2  2-argument predicate which can use the expanded tuple elements as arguments 
   * @return 8-argument predicate which returns the test result of the tuple returned by {@code fn1}
   * @param <S1> first argument of function and returned consumer/procedure
   * @param <S2> second argument of function and returned consumer/procedure
   * @param <S3> third argument of function and returned consumer/procedure
   * @param <S4> fourth argument of function and returned consumer/procedure
   * @param <S5> fifth argument of function and returned consumer/procedure
   * @param <S6> sixth argument of function and returned consumer/procedure
   * @param <S7> seventh argument of function and returned consumer/procedure
   * @param <S8> eighth argument of function and returned consumer/procedure
   * @param <E1> type of first element of intermediate tuple 
   * @param <E2> type of second element of intermediate tuple 
   */
  @NotNull
  static <S1, S2, S3, S4, S5, S6, S7, S8, E1, E2>
  Predicate8<S1, S2, S3, S4, S5, S6, S7, S8> concatPred8(@NotNull Function8<? extends ITuple2<? extends E1, ? extends E2>, S1, S2, S3, S4, S5, S6, S7, S8> fn1,
                                                         @NotNull BiPredicate<? super E1, ? super E2> pred2)
  {
    return (v1, v2, v3, v4, v5, v6, v7, v8) -> fn1.apply(v1, v2, v3, v4, v5, v6, v7, v8).testBy(pred2);
  }

  /**
   * Abstract base implementation of a 2-tuple.
   * This provides standard implementations for {@code java.lang.Object} methods
   * {@code equals()}, {@code hashCode()} and {@code toString()}. Therefore it is
   * most useful if you implement a 2-tuple on the fly.
   * @param <T1> type of first element
   * @param <T2> type of second element
   */
  abstract class Base<T1, T2> implements ITuple2<T1, T2>
  {
    @NotNull
    @Override
    public Base<T1, T2> asBase()
    {
      return this;
    }

    @Override
    public int hashCode()
    {
      return ITuple2.hash(this);
    }

    @Override
    public boolean equals(Object obj)
    {
      return ITuple2.equals(this, obj);
    }

    @Override
    public String toString()
    {
      return ITuple2.toString(this);
    }
  }
}
