// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.range;

import de.caff.annotation.NotNull;
import de.caff.generics.*;

import java.util.Collection;
import java.util.NoSuchElementException;
import java.util.PrimitiveIterator;

/**
 * Static class for providing ranges.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since March 02, 2020
 */
public class Range
{
  /** Don't construct. */
  private Range() { }

  /**
   * Get a range of integer numbers.
   * The returned range will return a sequence of numbers starting
   * with {@code start}, adding {@code step} on each iteration
   * until either {@code end} is reached or the following step
   * would reach beyond {@code end} (depending on step).
   * <p>
   * So {@code of(1, 4, 1)} would return {@code 1, 2, 3, 4)}, and
   * {@code of(1, 4, 2}} would return {@code 1, 3}.
   *
   * @param start start value of range (included)
   * @param end   end value of range (possibly included depending on
   *              {@code step}). Has to be the same or larger
   *              if {@code step} is positive, and same or smaller
   *              of {@code step} is negative
   * @param step  step size, must not be {@code 0}
   * @return lazy iterable of the given range
   * @throws IllegalArgumentException if {@code step} is {@code 0}
   *         or {@code start}, {@code end} and {@code step} do not match
   */
  @NotNull
  public static PrimitiveIntIterable of(int start, int end, int step)
  {
    if (step == 0) {
      throw new IllegalArgumentException("Not a valid step size: 0");
    }
    if (end < start && step > 0) {
      throw new IllegalArgumentException(String.format("Cannot step forward through backward range: %d..%d with step %d", start, end, step));
    }
    if (end > start && step < 0) {
      throw new IllegalArgumentException(String.format("Cannot step backward through forward range: %d..%d with step %d", start, end, step));
    }

    // things are more complex because of boundary conditions
    final int lastValid = Math.abs(step) == 1
            ? end
            : (int)((((long)end - (long)start) / step) * step) + start;
    return () -> new PrimitiveIterator.OfInt()
    {
      private int next = start;
      boolean running = true;

      @Override
      public int nextInt()
      {
        if (!running) {
          throw new NoSuchElementException("Iterator was already at its end!");
        }
        final int current = next;
        if (current == lastValid) {
          running = false;
        }
        else {
          next += step;
        }
        return current;
      }

      @Override
      public boolean hasNext()
      {
        return running;
      }
    };
  }

  /**
   * Get a range of integer numbers.
   * The returned range will return a sequence of numbers starting
   * with {@code start}, and a step width of either {@code 1}
   * (if {@code end >= start) or {@code -1}
   * (if {@code end < start}}.
   * @param start start value of range (included)
   * @param end   end value of range (included)
   * @return lazy iterable of the given range
   */
  @NotNull
  public static PrimitiveIntIterable of(int start, int end)
  {
    return start <= end
            ? of(start, end, 1)
            : of(start, end, -1);
  }

  /**
   * Get a range of long integer numbers.
   * The returned range will return a sequence of numbers starting
   * with {@code start}, adding {@code step} on each iteration
   * until either {@code end} is reached or the following step
   * would reach beyond {@code end} (depending on step).
   * <p>
   * So {@code of(1L, 4L, 1L)} would return {@code 1L, 2L, 3L, 4L)}, and
   * {@code of(1L, 4L, 2L}} would return {@code 1L, 3L}.
   *
   * @param start start value of range (included)
   * @param end   end value of range (possibly included depending on
   *              {@code step}). Has to be the same or larger
   *              if {@code step} is positive, and same or smaller
   *              of {@code step} is negative
   * @param step  step size, must not be {@code 0}
   * @return lazy iterable of the given range
   * @throws IllegalArgumentException if {@code step} is {@code 0}
   *         or {@code start}, {@code end} and {@code step} do not match
   */
  @NotNull
  public static PrimitiveLongIterable of(long start, long end, long step)
  {
    if (step == 0) {
      throw new IllegalArgumentException("Not a valid step size: 0");
    }
    if (end < start && step > 0) {
      throw new IllegalArgumentException(String.format("Cannot step forward through backward range: %d..%d with step %d", start, end, step));
    }
    if (end > start && step < 0) {
      throw new IllegalArgumentException(String.format("Cannot step backward through forward range: %d..%d with step %d", start, end, step));
    }

    // things are more complex because of boundary conditions
    final long lastValid;
    if (Math.abs(step) == 1) {
      lastValid = end;
    }
    else if (step > 0) {
      if (start < 0  &&  end >= 0) {
        final long offset = (-1L - start) % step;
        final long next = -1L + step - offset;

        if (offset > end) {
          lastValid = -1L - offset;
        }
        else {
          lastValid = (((end - next) / step) * step) + next;
        }
      }
      else {
        lastValid = (((end - start) / step) * step) + start;
      }
    }
    else {
      if (end < 0  &&  start >= 0) {
        final long offset = start % step;
        final long next = offset + step;
        if (next < end) {
          lastValid = offset;
        }
        else {
          lastValid = (((end - next) / step) * step) + next;
        }
      }
      else {
        lastValid = (((end - start) / step) * step) + start;
      }
    }
    return () -> new PrimitiveIterator.OfLong()
    {
      private long next = start;
      private boolean running = true;

      @Override
      public long nextLong()
      {
        if (!running) {
          throw new NoSuchElementException("Iterator was already at its end!");
        }
        final long current = next;
        if (current == lastValid) {
          running = false;
        }
        else {
          next += step;
        }
        return current;
      }

      @Override
      public boolean hasNext()
      {
        return running;
      }
    };
  }

  /**
   * Get a range of long integer numbers.
   * The returned range will provide a sequence of numbers starting
   * with {@code start}, and a step width of either {@code 1L}
   * (if {@code end >= start) or {@code -1L}
   * (if {@code end < start}}.
   * @param start start value of range (included)
   * @param end   end value of range (included)
   * @return lazy iterable of the given range
   */
  @NotNull
  public static PrimitiveLongIterable of(long start, long end)
  {
    return start <= end
            ? of(start, end, 1L)
            : of(start, end, -1L);
  }

  /**
   * Return indexes for accessing an array
   * or {@code java.util.List}.
   * @param length length/size of the array/list
   * @return range of indexes
   * @throws IllegalArgumentException if {@code length} is negative
   * @see de.caff.generics.IntIndexable#rangeFromSize(int) for an alternative
   */
  @NotNull
  public static PrimitiveIntIterable indexes(int length)
  {
    if (length < 0) {
      throw new IllegalArgumentException("Length must not be negative!");
    }
    return length != 0
            ? of(0, length - 1)
            : PrimitiveIntIterable.EMPTY;
  }

  /**
   * Return the range of indexes for accessing the given array.
   * @param array array
   * @return range of indexes
   */
  @NotNull
  public static PrimitiveIntIterable indexes(@NotNull Object[] array)
  {
    return indexes(array.length);
  }

  /**
   * Return the range of indexes for accessing the given list.
   * @param list list
   * @return range of indexes
   */
  @NotNull
  public static PrimitiveIntIterable indexes(@NotNull Collection<?> list)
  {
    return indexes(list.size());
  }

  /**
   * Return the range of indexes for accessing a sizeable object using standard indexing.
   * @param s sizeable object
   * @return range of indexes
   */
  @NotNull
  public static PrimitiveIntIterable indexes(@NotNull Sizeable s)
  {
    return indexes(s.size());
  }
}
