// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

/**
 * Multi-dimensional array access.
 * <p>
 * This package provides basic implementations
 * for multi-dimensional arrays. Internally
 * these arrays are handled as arrays (for
 * primitive types) or as {@link java.util.ArrayList}s,
 * <p>
 * Normally you'll either use {@link de.caff.generics.mda.MultiDimensionalArray}
 * for standard Java objects or one of the dedicated implementations for
 * primitive Java types:
 * <ul>
 *   <li>{@link de.caff.generics.mda.MultiDimensionalBooleanArray}</li>
 *   <li>{@link de.caff.generics.mda.MultiDimensionalByteArray}</li>
 *   <li>{@link de.caff.generics.mda.MultiDimensionalCharArray}</li>
 *   <li>{@link de.caff.generics.mda.MultiDimensionalDoubleArray}</li>
 *   <li>{@link de.caff.generics.mda.MultiDimensionalFloatArray}</li>
 *   <li>{@link de.caff.generics.mda.MultiDimensionalIntArray}</li>
 *   <li>{@link de.caff.generics.mda.MultiDimensionalLongArray}</li>
 *   <li>{@link de.caff.generics.mda.MultiDimensionalShortArray}</li>
 * </ul>
 *
 * The main advantages compared to using multiple standard arrays
 * like {@code int[][][]} are
 * <ul>
 *   <li>
 *     You can easily extract sub arrays with lower dimensionality
 *     and access these, even to manipulate the original array.
 *   </li>
 *   <li>The size of the single dimensions is clearly defined.</li>
 *   <li>The generic implementation is tyoe-safe.</li>
 * </ul>
 * The main disadvantages
 * <ul>
 *   <li>
 *     Access is some 10-30 times slower compared to raw arrays.
 *   </li>
 *   <li>
 *     You cannot have more that {@code Integer.MAX_VALUE} elements in
 *     the complete array, i.e. for all dimensions together..
 *   </li>
 * </ul>
 * If you only have 3 (or 2 or 1) dimensions you can avoid both disadvantages
 * by using one of the dedicated classes for 1-, 2- or 3-dimensional access:
 * <table border="1">
 *   <caption>Dedicated fast multi-dimensional arrays.</caption>
 *   <tr><th>1-dimensional</th><th>2-dimensional</th><th>3-dimensional</th><td>Usage</td></tr>
 *   <tr>
 *     <td>{@link de.caff.generics.mda.OneDimensionalArray}</td>
 *     <td>{@link de.caff.generics.mda.TwoDimensionalArray}</td>
 *     <td>{@link de.caff.generics.mda.ThreeDimensionalArray}</td>
 *     <td>for general types</td>
 *   </tr>
 *   <tr>
 *     <td>{@link de.caff.generics.mda.OneDimensionalBooleanArray}</td>
 *     <td>{@link de.caff.generics.mda.TwoDimensionalBooleanArray}</td>
 *     <td>{@link de.caff.generics.mda.ThreeDimensionalBooleanArray}</td>
 *     <td>for {@code boolean}</td>
 *   </tr>
 *   <tr>
 *     <td>{@link de.caff.generics.mda.OneDimensionalCharArray}</td>
 *     <td>{@link de.caff.generics.mda.TwoDimensionalCharArray}</td>
 *     <td>{@link de.caff.generics.mda.ThreeDimensionalCharArray}</td>
 *     <td>for {@code char}</td>
 *   </tr>
 *   <tr>
 *     <td>{@link de.caff.generics.mda.OneDimensionalDoubleArray}</td>
 *     <td>{@link de.caff.generics.mda.TwoDimensionalDoubleArray}</td>
 *     <td>{@link de.caff.generics.mda.ThreeDimensionalDoubleArray}</td>
 *     <td>for {@code double}</td>
 *   </tr>
 *   <tr>
 *     <td>{@link de.caff.generics.mda.OneDimensionalFloatArray}</td>
 *     <td>{@link de.caff.generics.mda.TwoDimensionalFloatArray}</td>
 *     <td>{@link de.caff.generics.mda.ThreeDimensionalFloatArray}</td>
 *     <td>for {@code float}</td>
 *   </tr>
 *   <tr>
 *     <td>{@link de.caff.generics.mda.OneDimensionalIntArray}</td>
 *     <td>{@link de.caff.generics.mda.TwoDimensionalIntArray}</td>
 *     <td>{@link de.caff.generics.mda.ThreeDimensionalIntArray}</td>
 *     <td>for {@code int}</td>
 *   </tr>
 *   <tr>
 *     <td>{@link de.caff.generics.mda.OneDimensionalLongArray}</td>
 *     <td>{@link de.caff.generics.mda.TwoDimensionalLongArray}</td>
 *     <td>{@link de.caff.generics.mda.ThreeDimensionalLongArray}</td>
 *     <td>for {@code long}</td>
 *   </tr>
 *   <tr>
 *     <td>{@link de.caff.generics.mda.OneDimensionalShortArray}</td>
 *     <td>{@link de.caff.generics.mda.TwoDimensionalShortArray}</td>
 *     <td>{@link de.caff.generics.mda.ThreeDimensionalShortArray}</td>
 *     <td>for {@code short}</td>
 *   </tr>
 * </table>
 * They provide access times comparable to raw arrays when using their
 * {@code getValueAt()}/–{@code setValueAt()} methods, but also fulfill the general
 * {@link de.caff.generics.mda.MultiDimensionalAccess} contracts. E.g.
 * the 2-dimensional classes allow access of rows or columns as
 * 1-dimensional arrays without the need of copying data.
 * <p>
 * Examples:
 * <pre>{@code
 *    import static de.caff.generics.tensor.MultiIndexLinearizer.OPEN;
 *    // [...]
 *    MultiDimensionalIntArray arr = new MultiDimensionalIntArray(4, 3, 2); // 3D
 *    arr.setFrom(0,       // 0, 0, 0
 *                1,       // 0. 0, 1
 *                2,       // 0, 1, 0
 *                3,       // 0, 1, 1
 *                4,       // 0, 2, 9
 *                5,       // 0, 2, 1
 *                6,       // 1, 0, 0
 *                7,       // 1, 0, 1
 *                8,       // 1, 1, 0
 *                9,       // 1, 1, 1
 *                10,      // 1, 2, 0
 *                11,      // 1, 2, 1
 *                12,      // 2, 0, 0
 *                13,      // 2, 0, 1
 *                14,      // 2, 1, 0
 *                15,      // 2, 1, 1
 *                16,      // 2, 2, 0
 *                17,      // 2, 2, 1
 *                18,      // 3, 0, 0
 *                19,      // 3, 0, 1
 *                20,      // 3. 1, 0
 *                21,      // 3, 1, 1
 *                22,      // 3, 2, 0
 *                23);     // 3, 2, 1
 *    int v = arr.get(2, 2, 1);  // v = 17
 *    MultiDimensionalIntArray sub1 = arr.sub(1); // sub1 is 2D (3x2) with the values 6,7, 8,9, 10,11
 *    MultiDimensionalIntArray sub2 = arr.sub(OPEN, 2); // sub2 is 2D (4x2) with 4,5, 10,11, 16,17, 22,23
 *    MultiDimensionalIntArray sub3 = arr.sub(OPEN, OPEN, 0); // sub3 is 2D (4x3) with all even values
 *    MultiDimensionalIntArray sub4 = arr.sub(3, 1); // sub4 is 1D (2) with 20, 21
 *    MultiDimensionalIntArray sub5 = arr.sub(3).sub(1); // sub5 is 1D (2) with 20. 21
 *    boolean test = sub4.equals(sub5);  // test is true
 *    MultiDimensionalIntArray sub6 = sub5.getCopy(); // sub6 is 1D (2) with 20, 21
 *    sub5.set(42, 1);  // In arrays arr, sub4, and sub5 the 21 changes to 42 because they are only views.
 *                      // In array sub6 the 21 stays untouched.
 * }</pre>
 */
package de.caff.generics.mda;