// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.IntToShortFunction2;
import de.caff.generics.function.ShortOperator1;

/**
 * Read/write access for 2-dimensional rectangular short arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface TwoDimensionalShortAccess
        extends TwoDimensionalShortReadAccess,
                TwoDimensionalAccess<Short>
{
  /** Empty 2dimensional short access. */
  TwoDimensionalShortAccess EMPTY = new TwoDimensionalShortAccess()
  {
    @Override
    public void setValueAt(short value, int ix, int iy)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public short getValueAt(int ix, int iy)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int sizeX()
    {
      return 0;
    }

    @Override
    public int sizeY()
    {
      return 0;
    }
  };

  /**
   * Set the value at the given indexes.
   * @param value  value to set
   * @param ix     X index
   * @param iy     Y index
   */
  void setValueAt(short value, int ix, int iy);

  /**
   * Change the value at the given indexes.
   * @param operator operator to apply to the value
   * @param ix       X index
   * @param iy       Y index
   */
  default void changeValueAt(@NotNull ShortOperator1 operator, int ix, int iy)
  {
    setValueAt(operator.applyAsShort(getValueAt(ix, iy)), ix, iy);
  }

  /**
   * Change all values by applying an operator.
   * @param operator operator to apply
   */
  default void changeAllValues(@NotNull ShortOperator1 operator)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        changeValueAt(operator, x, y);
      }
    }
  }

  /**
   * Set all values depending on their indexes.
   * This will call the setter for all indexes and set the value at the given index
   * from its result.
   * @param setter setter which provides the value to set for each index combination
   */
  default void fillValuesByIndex(@NotNull IntToShortFunction2 setter)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        setElementAt(setter.applyAsShort(x, y), x, y);
      }
    }
  }

  @Override
  default void setElementAt(@NotNull Short element, int ix, int iy)
  {
    setValueAt(element, ix, iy);
  }

  @NotNull
  @Override
  default OneDimensionalShortAccess subAtX(int ix)
  {
    return new OneDimensionalShortAccess()
    {
      @Override
      public void setValueAt(short value, int index)
      {
        TwoDimensionalShortAccess.this.setValueAt(value, ix, index);
      }

      @Override
      public short getValueAt(int index)
      {
        return TwoDimensionalShortAccess.this.getValueAt(ix, index);
      }

      @Override
      public int size()
      {
        return sizeY();
      }
    };
  }

  @NotNull
  @Override
  default OneDimensionalShortAccess subAtY(int iy)
  {
    return new OneDimensionalShortAccess()
    {
      @Override
      public void setValueAt(short value, int index)
      {
        TwoDimensionalShortAccess.this.setValueAt(value, index, iy);
      }

      @Override
      public short getValueAt(int index)
      {
        return TwoDimensionalShortAccess.this.getValueAt(index, iy);
      }

      @Override
      public int size()
      {
        return sizeX();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalShortAccess transposed()
  {
    return new TwoDimensionalShortAccess()
    {
      @Override
      public void setValueAt(short value, int ix, int iy)
      {
        TwoDimensionalShortAccess.this.setValueAt(value, iy, ix);
      }

      @Override
      public short getValueAt(int ix, int iy)
      {
        return TwoDimensionalShortAccess.this.getValueAt(iy, ix);
      }

      @Override
      public int sizeX()
      {
        return TwoDimensionalShortAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalShortAccess.this.sizeX();
      }
    };
  }
}
