// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.Countable;
import de.caff.generics.Types;
import de.caff.generics.function.Procedure1;
import de.caff.generics.function.Procedure2;

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.function.BiPredicate;
import java.util.function.Function;

/**
 * Read access for 2-dimensional rectangular arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface TwoDimensionalReadAccess<T>
        extends MultiDimensionalReadAccess<T>
{
  /** Empty acces, use {@link #empty()} instead. */
  TwoDimensionalReadAccess<?> EMPTY = new TwoDimensionalReadAccess<Object>()
  {
    @Override
    public int sizeX()
    {
      return 0;
    }

    @Override
    public int sizeY()
    {
      return 0;
    }

    @Override
    public Object getElementAt(int ix, int iy)
    {
      throw new IndexOutOfBoundsException();
    }
  };

  /**
   * Get the number of elements in X direction.
   * @return number of elements in X direction
   */
  int sizeX();

  /**
   * Get the number of elements in Y direction
   * @return number of elements in Y direction
   */
  int sizeY();

  /**
   * Get the element at the given indexes.
   * @param ix index in X direction from {@code 0} (included) to {@link #sizeX()} (excluded)
   * @param iy index in Y direction from {@code 0} (included) to {@link #sizeY()} (excluded)
   * @return element at the given indexes
   */
  T getElementAt(int ix, int iy);

  @Override
  default T getElement(int... indexes)
  {
    if (indexes.length != 2) {
      throw new IllegalArgumentException("Need 2 index for 2-dimensional array!");
    }
    return getElementAt(indexes[0], indexes[1]);
  }

  /**
   * Get a 1-dimensional view of the row at the given X index.
   * This is a view, so setting elements will change values in this array.
   * @param ix X index
   * @return 1-dimensional array at {@code ix}
   */
  @NotNull
  default OneDimensionalReadAccess<T> subAtX(int ix)
  {
    return new OneDimensionalReadAccess<T>()
    {
      @Override
      public int size()
      {
        return sizeY();
      }

      @Override
      public T get(int index)
      {
        return getElementAt(ix, index);
      }
    };
  }

  /**
   * Get an iterator which iterates over rows with cnnstant X.
   * @return X row iterator
   */
  @NotNull
  default Iterator<OneDimensionalReadAccess<T>> xIterator()
  {
    return new Iterator<OneDimensionalReadAccess<T>>()
    {
      private int x = 0;
      @Override
      public boolean hasNext()
      {
        return x < sizeX();
      }

      @Override
      public OneDimensionalReadAccess<T> next()
      {
        if (x >= sizeX()) {
          throw new NoSuchElementException();
        }
        return subAtX(x++);
      }
    };
  }

  /**
   * Get an iterable which iterates over rows with constant X.
   * @return X row iterable
   */
  @NotNull
  default Iterable<OneDimensionalReadAccess<T>> xIterable()
  {
    return this::xIterator;
  }

  /**
   * Get an iterator which iterates over rows with cnnstant Y.
   * @return Y row iterator
   */
  @NotNull
  default Iterator<OneDimensionalReadAccess<T>> yIterator()
  {
    return new Iterator<OneDimensionalReadAccess<T>>()
    {
      private int y = 0;
      @Override
      public boolean hasNext()
      {
        return y < sizeY();
      }

      @Override
      public OneDimensionalReadAccess<T> next()
      {
        if (y >= sizeY()) {
          throw new NoSuchElementException();
        }
        return subAtY(y++);
      }
    };
  }

  /**
   * Get an iterable which iterates over rows with constant Y.
   * @return Y row iterable
   */
  @NotNull
  default Iterable<OneDimensionalReadAccess<T>> yIterable()
  {
    return this::yIterator;
  }

  /**
   * Get a 1-dimensional view of the column at the given X index.
   * This is a view, so setting elements will change values in this array.
   * @param iy Y index
   * @return 1-dimensional array at {@code iy}
   */
  @NotNull
  default OneDimensionalReadAccess<T> subAtY(int iy)
  {
    return new OneDimensionalReadAccess<T>()
    {
      @Override
      public int size()
      {
        return sizeX();
      }

      @Override
      public T get(int index)
      {
        return getElementAt(index, iy);
      }
    };
  }

  /**
   * Get a transposed view of this 2-dimensional read access.
   * @return transposed view where X and Y are interchanged
   */
  @NotNull
  default TwoDimensionalReadAccess<T> transposed()
  {
    return new TwoDimensionalReadAccess<T>()
    {
      @Override
      public int sizeX()
      {
        return TwoDimensionalReadAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalReadAccess.this.sizeX();
      }

      @Override
      public T getElementAt(int ix, int iy)
      {
        return TwoDimensionalReadAccess.this.getElementAt(iy, ix);
      }

      @NotNull
      @Override
      public TwoDimensionalReadAccess<T> transposed()
      {
        return TwoDimensionalReadAccess.this;
      }
    };
  }

  @Override
  default int getNumDimensions()
  {
    return 2;
  }

  @Override
  default int getSize(int dim)
  {
    switch (dim) {
    case 0:
      return sizeX();
    case 1:
      return sizeY();
    }
    throw new IllegalArgumentException("There are only 2 dimensions!");
  }

  @NotNull
  @Override
  default int[] getSizes()
  {
    return new int[] { sizeX(), sizeY() };
  }

  @Override
  default long getNumElements()
  {
    return (long)sizeX() * sizeY();
  }

  /**
   * Get a view of this two dimensional array with converted elements.
   * @param conv   element converter
   * @param <TOUT>  element type provided by the view
   * @return view of this array with converted elements
   */
  @NotNull
  default <TOUT> TwoDimensionalReadAccess<TOUT> view(@NotNull Function<T, TOUT> conv)
  {
    return new TwoDimensionalReadAccess<TOUT>()
    {
      @Override
      public int sizeX()
      {
        return TwoDimensionalReadAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalReadAccess.this.sizeY();
      }

      @Override
      public TOUT getElementAt(int ix, int iy)
      {
        return conv.apply(TwoDimensionalReadAccess.this.getElementAt(ix, iy));
      }
    };
  }

  @Override
  default void visitAll(@NotNull Procedure2<? super T, int[]> visitor)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        visitor.apply(getElementAt(x, y), new int[] { x , y });
      }
    }
  }

  @Override
  default void visitAll(@NotNull Procedure1<? super T> visitor)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        visitor.apply(getElementAt(x, y));
      }
    }
  }

  /**
   * View this class as if it has useful defaults for standard object methods.
   * @return based view of this class
   */
  @NotNull
  default Base<T> asBase()
  {
    return new Base<T>()
    {
      @Override
      public int sizeX()
      {
        return TwoDimensionalReadAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalReadAccess.this.sizeY();
      }

      @Override
      public T getElementAt(int ix, int iy)
      {
        return TwoDimensionalReadAccess.this.getElementAt(ix, iy);
      }
    };
  }

  /**
   * Create a string representation of an 2-dimensional read access.
   * @param access access to stringify
   * @return string representation
   */
  @NotNull
  static String toString(@NotNull TwoDimensionalReadAccess<?> access)
  {
    final StringBuilder sb = new StringBuilder();
    final int sx = access.sizeX();
    final int sy = access.sizeY();
    sb.append("[");
    for (int x = 0;  x < sx;  ++x) {
      sb.append(x == 0 ? "[" : ", [");
      for (int y = 0;  y < sy;  ++y) {
        if (y > 0) {
          sb.append(", ");
        }
        sb.append(access.getElementAt(x, y));
      }
      sb.append("]");
    }
    sb.append("]");
    return sb.toString();
  }

  /**
   * Are two 2-dimensional accesses equal?
   *
   * @param access1 first access
   * @param access2 second access
   * @param equalityChecker checker for equality of the elements of both objects
   * @param <E> parameter type of equality checker, and super class of
   *           both accesses element type
   * @return {@code true} if both accesses contain the same values in the same sequence<br>
   *         {@code false} if sizes or values differ
   */
  static <E> boolean equal(@NotNull TwoDimensionalReadAccess<? extends E> access1,
                           @NotNull TwoDimensionalReadAccess<? extends E> access2,
                           @NotNull BiPredicate<E, E> equalityChecker)
  {
    if (access1 == access2) {
      return true;
    }
    if (access1.sizeX() != access2.sizeX()  ||
        access1.sizeY() != access2.sizeY()) {
      return false;
    }
    final Iterator<? extends OneDimensionalReadAccess<? extends E>> it1 = access1.xIterator();
    final Iterator<? extends OneDimensionalReadAccess<? extends E>> it2 = access2.xIterator();
    while (it1.hasNext()  &&  it2.hasNext()) {
      if (!Countable.equal(it1.next(), it2.next(), equalityChecker)) {
        return false;
      }
    }
    return !(it1.hasNext()  ||  it2.hasNext());  // should always evaluate to true
  }

  /**
   * Calculate a hashcode for a 2-dimensional read access..
   * @param access access for which the hash code is required
   * @return hash code for the given access
   */
  static int hash(@NotNull TwoDimensionalReadAccess<?> access)
  {
    // this follows Arrays.hashCode()
    int result = 1;

    for (OneDimensionalReadAccess<?> row : access.xIterable()) {
      result = 31 * result + Types.hash(row);
    }

    return result;
  }

  /**
   * Get an empty 2dimensional read access.
   * @param <E> element type
   * @return {@link #EMPTY} in a form the compiler accepts
   */
  @NotNull
  @SuppressWarnings("unchecked") // as EMPTY is immutable
  static <E> TwoDimensionalReadAccess<E> empty()
  {
    return (TwoDimensionalReadAccess<E>)EMPTY;
  }

  /**
   * Get a 2dimensional read access for a single element.
   * @param singleElement single element
   * @return 2dimensional read access with size 1 in both directions,
   *         containing just the given single element
   * @param <E> element type
   */
  @NotNull
  static <E> TwoDimensionalReadAccess<E> singleton(E singleElement)
  {
    return new Base<E>()
    {
      @Override
      public int sizeX()
      {
        return 1;
      }

      @Override
      public int sizeY()
      {
        return 1;
      }

      @Override
      public E getElementAt(int ix, int iy)
      {
        if (ix != 0  ||  iy != 0) {
          throw new IndexOutOfBoundsException();
        }
        return singleElement;
      }
    };
  }

  /**
   * Base class implementing this interface which provides useful
   * default methods for {@link Object#toString()}, {@link Object#equals(Object)},
   * and {@link Object#hashCode()}.
   * @param <TT> element type
   */
  abstract class Base<TT> implements TwoDimensionalReadAccess<TT>
  {
    @Override
    public String toString()
    {
      return TwoDimensionalReadAccess.toString(this);
    }

    @Override
    @SuppressWarnings("unchecked")
    public boolean equals(Object obj)
    {
      if (!(obj instanceof TwoDimensionalReadAccess)) {
        return false;
      }
      return equal(this, (TwoDimensionalReadAccess<? extends TT>)obj, Objects::deepEquals);
    }

    @Override
    public int hashCode()
    {
      return hash(this);
    }

    @NotNull
    @Override
    public Base<TT> asBase()
    {
      return this;
    }
  }
}
