// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;

/**
 * Read access for 2-dimensional rectangular long arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface TwoDimensionalLongReadAccess
        extends TwoDimensionalReadAccess<Long>
{
  /**
   * Get the value at the given indexes.
   * @param ix X index
   * @param iy Y index
   * @return value at the given index
   */
  long getValueAt(int ix, int iy);

  @Override
  @NotNull
  default Long getElementAt(int ix, int iy)
  {
    return getValueAt(ix, iy);
  }

  @NotNull
  @Override
  default OneDimensionalLongReadAccess subAtX(int ix)
  {
    return new OneDimensionalLongReadAccess()
    {
      @Override
      public long getValueAt(int index)
      {
        return TwoDimensionalLongReadAccess.this.getValueAt(ix, index);
      }

      @Override
      public int size()
      {
        return sizeY();
      }
    };
  }

  @NotNull
  @Override
  default OneDimensionalLongReadAccess subAtY(int iy)
  {
    return new OneDimensionalLongReadAccess()
    {
      @Override
      public long getValueAt(int index)
      {
        return TwoDimensionalLongReadAccess.this.getValueAt(index, iy);
      }

      @Override
      public int size()
      {
        return sizeX();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalLongReadAccess transposed()
  {
    return new TwoDimensionalLongReadAccess()
    {
      @Override
      public long getValueAt(int ix, int iy)
      {
        return TwoDimensionalLongReadAccess.this.getValueAt(iy, ix);
      }

      @Override
      public int sizeX()
      {
        return TwoDimensionalLongReadAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalLongReadAccess.this.sizeX();
      }
    };
  }

  /**
   * Get a view of this 2-dimensional access which provides useful defaults for standard object methods.
   * @return base view of this object
   */
  @NotNull
  default TwoDimensionalLongReadAccess.Base asBase()
  {
    return new TwoDimensionalLongReadAccess.Base()
    {
      @Override
      public long getValueAt(int ix, int iy)
      {
        return TwoDimensionalLongReadAccess.this.getValueAt(ix, iy);
      }

      @Override
      public int sizeX()
      {
        return TwoDimensionalLongReadAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalLongReadAccess.this.sizeX();
      }
    };
  }

  /**
   * Base class implementing this interface which provides useful
   * default methods for {@link Object#toString()}, {@link Object#equals(Object)},
   * and {@link Object#hashCode()}.
   */
  abstract class Base extends TwoDimensionalReadAccess.Base<Long>
          implements TwoDimensionalLongReadAccess
  {
    @NotNull
    @Override
    public TwoDimensionalLongReadAccess.Base asBase()
    {
      return this;
    }
  }
}
