// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;

/**
 * A two-dimensional array of long values.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public class TwoDimensionalLongArray
        extends TwoDimensionalLongReadAccess.Base
        implements TwoDimensionalLongAccess
{
  @NotNull
  private final long[][] values;

  /**
   * Constructor.
   * @param xSize X size of array
   * @param ySize Y size of array
   */
  public TwoDimensionalLongArray(int xSize, int ySize)
  {
    values = new long[xSize][ySize];
  }

  /**
   * Constructor.
   * @param ySize  size in y direction
   * @param values values in order, the first {@code ySize} values
   *               will become row 0, the second {@code ySize} values
   *               row 1, and so on. Needs a multiple of ySize values.
   */
  public TwoDimensionalLongArray(int ySize,
                                 long... values)
  {
    if (ySize <= 0  ||  values.length == 0) {
      throw new IllegalArgumentException("Need positive sizes!");
    }
    if (values.length % ySize != 0) {
      throw new IllegalArgumentException("Need a multiple of ySize values!");
    }
    this.values = new long[values.length / ySize][ySize];
    for (int v = 0;  v < values.length;  ++v) {
      this.values[v / ySize][v % ySize] = values[v];
    }
  }

  /**
   * Set the value at the given indexes.
   * @param value  value to set
   * @param ix     X index
   * @param iy     Y index
   */
  public void setValueAt(long value, int ix, int iy)
  {
    values[ix][iy] = value;
  }

  /**
   * Get the value at the given indexes.
   * @param ix  X index
   * @param iy  Y index
   * @return value at given index
   */
  public long getValueAt(int ix, int iy)
  {
    return values[ix][iy];
  }

  @Override
  public int sizeX()
  {
    return values.length;
  }

  @Override
  public int sizeY()
  {
    return values[0].length;
  }
}
