// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.IntToLongFunction2;
import de.caff.generics.function.LongOperator1;

/**
 * Read/write access for 2-dimensional rectangular long arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface TwoDimensionalLongAccess
        extends TwoDimensionalLongReadAccess,
                TwoDimensionalAccess<Long>
{

  /**
   * Set the value at the given indexes.
   * @param value  value to set
   * @param ix     X index
   * @param iy     Y index
   */
  void setValueAt(long value, int ix, int iy);

  /**
   * Change the value at the given indexes.
   * @param operator operator to apply to the value
   * @param ix       X index
   * @param iy       Y index
   */
  default void changeValueAt(@NotNull LongOperator1 operator, int ix, int iy)
  {
    setValueAt(operator.applyAsLong(getValueAt(ix, iy)), ix, iy);
  }

  /**
   * Change all values by applying an operator.
   * @param operator operator to apply
   */
  default void changeAllValues(@NotNull LongOperator1 operator)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        changeValueAt(operator, x, y);
      }
    }
  }

  /**
   * Set all values depending on their indexes.
   * This will call the setter for all indexes and set the value at the given index
   * from its result.
   * @param setter setter which provides the value to set for each index combination
   */
  default void fillValuesByIndex(@NotNull IntToLongFunction2 setter)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        setElementAt(setter.applyAsLong(x, y), x, y);
      }
    }
  }

  @Override
  default void setElementAt(@NotNull Long element, int ix, int iy)
  {
    setValueAt(element, ix, iy);
  }

  @NotNull
  @Override
  default OneDimensionalLongAccess subAtX(int ix)
  {
    return new OneDimensionalLongAccess()
    {
      @Override
      public void setValueAt(long value, int index)
      {
        TwoDimensionalLongAccess.this.setValueAt(value, ix, index);
      }

      @Override
      public long getValueAt(int index)
      {
        return TwoDimensionalLongAccess.this.getValueAt(ix, index);
      }

      @Override
      public int size()
      {
        return sizeY();
      }
    };
  }

  @NotNull
  @Override
  default OneDimensionalLongAccess subAtY(int iy)
  {
    return new OneDimensionalLongAccess()
    {
      @Override
      public void setValueAt(long value, int index)
      {
        TwoDimensionalLongAccess.this.setValueAt(value, index, iy);
      }

      @Override
      public long getValueAt(int index)
      {
        return TwoDimensionalLongAccess.this.getValueAt(index, iy);
      }

      @Override
      public int size()
      {
        return sizeX();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalLongAccess transposed()
  {
    return new TwoDimensionalLongAccess()
    {
      @Override
      public void setValueAt(long value, int ix, int iy)
      {
        TwoDimensionalLongAccess.this.setValueAt(value, iy, ix);
      }

      @Override
      public long getValueAt(int ix, int iy)
      {
        return TwoDimensionalLongAccess.this.getValueAt(iy, ix);
      }

      @Override
      public int sizeX()
      {
        return TwoDimensionalLongAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalLongAccess.this.sizeX();
      }
    };
  }
}
