// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;

/**
 * Read access for 2-dimensional rectangular int arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface TwoDimensionalIntReadAccess
        extends TwoDimensionalReadAccess<Integer>
{
  /** Empty 2dimensional int read access. */
  TwoDimensionalIntReadAccess EMPTY = new TwoDimensionalIntReadAccess.Base()
  {
    @Override
    public int getValueAt(int ix, int iy)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int sizeX()
    {
      return 0;
    }

    @Override
    public int sizeY()
    {
      return 0;
    }
  };

  /**
   * Get the value at the given indexes.
   * @param ix X index
   * @param iy Y index
   * @return value at the given index
   */
  int getValueAt(int ix, int iy);

  @Override
  @NotNull
  default Integer getElementAt(int ix, int iy)
  {
    return getValueAt(ix, iy);
  }

  @NotNull
  @Override
  default OneDimensionalIntReadAccess subAtX(int ix)
  {
    return new OneDimensionalIntReadAccess()
    {
      @Override
      public int getValueAt(int index)
      {
        return TwoDimensionalIntReadAccess.this.getValueAt(ix, index);
      }

      @Override
      public int size()
      {
        return sizeY();
      }
    };
  }

  @NotNull
  @Override
  default OneDimensionalIntReadAccess subAtY(int iy)
  {
    return new OneDimensionalIntReadAccess()
    {
      @Override
      public int getValueAt(int index)
      {
        return TwoDimensionalIntReadAccess.this.getValueAt(index, iy);
      }

      @Override
      public int size()
      {
        return sizeX();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalIntReadAccess transposed()
  {
    return new TwoDimensionalIntReadAccess()
    {
      @Override
      public int getValueAt(int ix, int iy)
      {
        return TwoDimensionalIntReadAccess.this.getValueAt(iy, ix);
      }

      @Override
      public int sizeX()
      {
        return TwoDimensionalIntReadAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalIntReadAccess.this.sizeX();
      }
    };
  }

  /**
   * Get a view of this 2-dimensional access which provides useful defaults for standard object methods.
   * @return base view of this object
   */
  @NotNull
  default TwoDimensionalIntReadAccess.Base asBase()
  {
    return new TwoDimensionalIntReadAccess.Base()
    {
      @Override
      public int getValueAt(int ix, int iy)
      {
        return TwoDimensionalIntReadAccess.this.getValueAt(ix, iy);
      }

      @Override
      public int sizeX()
      {
        return TwoDimensionalIntReadAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalIntReadAccess.this.sizeX();
      }
    };
  }

  /**
   * Get a 2dimensional int read access for a single element.
   * @param singleElement single element
   * @return 2dimensional int read access with size 1 in both directions,
   *         containing just the given single element
   */
  @NotNull
  static TwoDimensionalIntReadAccess singleton(int singleElement)
  {
    return new TwoDimensionalIntReadAccess.Base()
    {
      @Override
      public int getValueAt(int ix, int iy)
      {
        if (ix != 0  ||  iy != 0) {
          throw new IndexOutOfBoundsException();
        }
        return singleElement;
      }

      @Override
      public int sizeX()
      {
        return 1;
      }

      @Override
      public int sizeY()
      {
        return 1;
      }
    };
  }

  /**
   * Base class implementing this interface which provides useful
   * default methods for {@link Object#toString()}, {@link Object#equals(Object)},
   * and {@link Object#hashCode()}.
   */
  abstract class Base extends TwoDimensionalReadAccess.Base<Integer>
          implements TwoDimensionalIntReadAccess
  {
    @NotNull
    @Override
    public TwoDimensionalIntReadAccess.Base asBase()
    {
      return this;
    }
  }
}
