// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.IntToFloatFunction2;

/**
 * A two-dimensional array of float values.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public class TwoDimensionalFloatArray
        extends TwoDimensionalFloatReadAccess.Base
        implements TwoDimensionalFloatAccess
{
  @NotNull
  private final float[][] values;

  /**
   * Constructor.
   * This fills all elements with {@code 0.0F} values.
   * @param xSize X size of array
   * @param ySize Y size of array
   */
  public TwoDimensionalFloatArray(int xSize, int ySize)
  {
    values = new float[xSize][ySize];
  }

  /**
   * Initializing constructor.
   * @param xSize X size of array
   * @param ySize Y size of array
   * @param filler setter function which provides the value to set for each index combination
   * @see #fillValuesByIndex(IntToFloatFunction2)
   */
  public TwoDimensionalFloatArray(int xSize, int ySize, @NotNull IntToFloatFunction2 filler)
  {
    this(xSize, ySize);
    fillValuesByIndex(filler);
  }

  /**
   * Constructor.
   * @param ySize  size in y direction
   * @param values values in order, the first {@code ySize} values
   *               will become row 0, the second {@code ySize} values
   *               row 1, and so on. Needs a multiple of ySize values.
   */
  public TwoDimensionalFloatArray(int ySize,
                                  float... values)
  {
    if (ySize <= 0  ||  values.length == 0) {
      throw new IllegalArgumentException("Need positive sizes!");
    }
    if (values.length % ySize != 0) {
      throw new IllegalArgumentException("Need a multiple of ySize values!");
    }
    this.values = new float[values.length / ySize][ySize];
    for (int v = 0;  v < values.length;  ++v) {
      this.values[v / ySize][v % ySize] = values[v];
    }
  }

  /**
   * Set the value at the given indexes.
   * @param value  value to set
   * @param ix     X index
   * @param iy     Y index
   */
  public void setValueAt(float value, int ix, int iy)
  {
    values[ix][iy] = value;
  }

  /**
   * Get the value at the given indexes.
   * @param ix  X index
   * @param iy  Y index
   * @return value at given index
   */
  public float getValueAt(int ix, int iy)
  {
    return values[ix][iy];
  }

  @Override
  public int sizeX()
  {
    return values.length;
  }

  @Override
  public int sizeY()
  {
    return values[0].length;
  }
}
