// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;

/**
 * Read access for 2-dimensional rectangular byte arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface TwoDimensionalByteReadAccess
        extends TwoDimensionalReadAccess<Byte>
{
  /** Empty 2dimensional byte read access. */
  TwoDimensionalByteReadAccess EMPTY = new TwoDimensionalByteReadAccess.Base()
  {
    @Override
    public byte getValueAt(int ix, int iy)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int sizeX()
    {
      return 0;
    }

    @Override
    public int sizeY()
    {
      return 0;
    }
  };

  /**
   * Get the value at the given indexes.
   * @param ix X index
   * @param iy Y index
   * @return value at the given index
   */
  byte getValueAt(int ix, int iy);

  @Override
  @NotNull
  default Byte getElementAt(int ix, int iy)
  {
    return getValueAt(ix, iy);
  }

  @NotNull
  @Override
  default OneDimensionalByteReadAccess subAtX(int ix)
  {
    return new OneDimensionalByteReadAccess()
    {
      @Override
      public byte getValueAt(int index)
      {
        return TwoDimensionalByteReadAccess.this.getValueAt(ix, index);
      }

      @Override
      public int size()
      {
        return sizeY();
      }
    };
  }

  @NotNull
  @Override
  default OneDimensionalByteReadAccess subAtY(int iy)
  {
    return new OneDimensionalByteReadAccess()
    {
      @Override
      public byte getValueAt(int index)
      {
        return TwoDimensionalByteReadAccess.this.getValueAt(index, iy);
      }

      @Override
      public int size()
      {
        return sizeX();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalByteReadAccess transposed()
  {
    return new TwoDimensionalByteReadAccess()
    {
      @Override
      public byte getValueAt(int ix, int iy)
      {
        return TwoDimensionalByteReadAccess.this.getValueAt(iy, ix);
      }

      @Override
      public int sizeX()
      {
        return TwoDimensionalByteReadAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalByteReadAccess.this.sizeX();
      }
    };
  }

  /**
   * Get a view of this 2-dimensional access which provides useful defaults for standard object methods.
   * @return base view of this object
   */
  @NotNull
  default TwoDimensionalByteReadAccess.Base asBase()
  {
    return new TwoDimensionalByteReadAccess.Base()
    {
      @Override
      public byte getValueAt(int ix, int iy)
      {
        return TwoDimensionalByteReadAccess.this.getValueAt(ix, iy);
      }

      @Override
      public int sizeX()
      {
        return TwoDimensionalByteReadAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalByteReadAccess.this.sizeX();
      }
    };
  }

  /**
   * Get a 2dimensional byte read access for a single element.
   * @param singleElement single element
   * @return 2dimensional byte read access with size 1 in both directions,
   *         containing just the given single element
   */
  @NotNull
  static TwoDimensionalByteReadAccess singleton(byte singleElement)
  {
    return new TwoDimensionalByteReadAccess.Base()
    {
      @Override
      public byte getValueAt(int ix, int iy)
      {
        if (ix != 0  ||  iy != 0) {
          throw new IndexOutOfBoundsException();
        }
        return singleElement;
      }

      @Override
      public int sizeX()
      {
        return 1;
      }

      @Override
      public int sizeY()
      {
        return 1;
      }
    };
  }

  /**
   * Base class implementing this interface which provides useful
   * default methods for {@link Object#toString()}, {@link Object#equals(Object)},
   * and {@link Object#hashCode()}.
   */
  abstract class Base extends TwoDimensionalReadAccess.Base<Byte>
          implements TwoDimensionalByteReadAccess
  {
    @NotNull
    @Override
    public TwoDimensionalByteReadAccess.Base asBase()
    {
      return this;
    }
  }
}
