// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;

/**
 * Read access for 2-dimensional rectangular boolean arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface TwoDimensionalBooleanReadAccess
        extends TwoDimensionalReadAccess<Boolean>
{
  /** Empty 2dimensional boolean read access. */
  TwoDimensionalBooleanReadAccess EMPTY = new TwoDimensionalBooleanReadAccess.Base()
  {
    @Override
    public boolean getValueAt(int ix, int iy)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int sizeX()
    {
      return 0;
    }

    @Override
    public int sizeY()
    {
      return 0;
    }
  };

  /**
   * Get the value at the given indexes.
   * @param ix X index
   * @param iy Y index
   * @return value at the given index
   */
  boolean getValueAt(int ix, int iy);

  @Override
  @NotNull
  default Boolean getElementAt(int ix, int iy)
  {
    return getValueAt(ix, iy);
  }

  @NotNull
  @Override
  default OneDimensionalBooleanReadAccess subAtX(int ix)
  {
    return new OneDimensionalBooleanReadAccess()
    {
      @Override
      public boolean getValueAt(int index)
      {
        return TwoDimensionalBooleanReadAccess.this.getValueAt(ix, index);
      }

      @Override
      public int size()
      {
        return sizeY();
      }
    };
  }

  @NotNull
  @Override
  default OneDimensionalBooleanReadAccess subAtY(int iy)
  {
    return new OneDimensionalBooleanReadAccess.Base()
    {
      @Override
      public boolean getValueAt(int index)
      {
        return TwoDimensionalBooleanReadAccess.this.getValueAt(index, iy);
      }

      @Override
      public int size()
      {
        return sizeX();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalBooleanReadAccess transposed()
  {
    return new TwoDimensionalBooleanReadAccess.Base()
    {
      @Override
      public boolean getValueAt(int ix, int iy)
      {
        return TwoDimensionalBooleanReadAccess.this.getValueAt(iy, ix);
      }

      @Override
      public int sizeX()
      {
        return TwoDimensionalBooleanReadAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalBooleanReadAccess.this.sizeX();
      }
    };
  }

  /**
   * Get a view of this 2-dimensional access which provides useful defaults for standard object methods.
   * @return base view of this object
   */
  @NotNull
  default TwoDimensionalBooleanReadAccess.Base asBase()
  {
    return new Base()
    {
      @Override
      public boolean getValueAt(int ix, int iy)
      {
        return TwoDimensionalBooleanReadAccess.this.getValueAt(ix, iy);
      }

      @Override
      public int sizeX()
      {
        return TwoDimensionalBooleanReadAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalBooleanReadAccess.this.sizeX();
      }
    };
  }

  /**
   * Get a 2dimensional boolean read access for a single element.
   * @param singleElement single element
   * @return 2dimensional boolean read access with size 1 in both directions,
   *         containing just the given single element
   */
  @NotNull
  static TwoDimensionalBooleanReadAccess singleton(boolean singleElement)
  {
    return new TwoDimensionalBooleanReadAccess.Base()
    {
      @Override
      public boolean getValueAt(int ix, int iy)
      {
        if (ix != 0  ||  iy != 0) {
          throw new IndexOutOfBoundsException();
        }
        return singleElement;
      }

      @Override
      public int sizeX()
      {
        return 1;
      }

      @Override
      public int sizeY()
      {
        return 1;
      }
    };
  }

  /**
   * Base class implementing this interface which provides useful
   * default methods for {@link Object#toString()}, {@link Object#equals(Object)},
   * and {@link Object#hashCode()}.
   */
  abstract class Base extends TwoDimensionalReadAccess.Base<Boolean>
          implements TwoDimensionalBooleanReadAccess
  {
    @NotNull
    @Override
    public TwoDimensionalBooleanReadAccess.Base asBase()
    {
      return this;
    }
  }
}
