// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.BooleanOperator1;
import de.caff.generics.function.IntPredicate2;

/**
 * Read/write access for 2-dimensional rectangular boolean arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface TwoDimensionalBooleanAccess
        extends TwoDimensionalBooleanReadAccess,
                TwoDimensionalAccess<Boolean>
{
  /** Empty 2dimensional boolean access. */
  TwoDimensionalBooleanAccess EMPTY = new TwoDimensionalBooleanAccess()
  {
    @Override
    public void setValueAt(boolean value, int ix, int iy)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public boolean getValueAt(int ix, int iy)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int sizeX()
    {
      return 0;
    }

    @Override
    public int sizeY()
    {
      return 0;
    }
  };

  /**
   * Set the value at the given indexes.
   * @param value  value to set
   * @param ix     X index
   * @param iy     Y index
   */
  void setValueAt(boolean value, int ix, int iy);

  /**
   * Change the value at the given indexes.
   * @param operator operator to apply to the value
   * @param ix       X index
   * @param iy       Y index
   */
  default void changeValueAt(@NotNull BooleanOperator1 operator, int ix, int iy)
  {
    setValueAt(operator.applyAsBoolean(getValueAt(ix, iy)), ix, iy);
  }

  /**
   * Change all values by applying an operator.
   * @param operator operator to apply
   */
  default void changeAllValues(@NotNull BooleanOperator1 operator)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        changeValueAt(operator, x, y);
      }
    }
  }

  /**
   * Set all values depending on their indexes.
   * This will call the setter for all indexes and set the value at the given index
   * from its result.
   * @param setter setter which provides the value to set for each index combination
   */
  default void fillValuesByIndex(@NotNull IntPredicate2 setter)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        setElementAt(setter.testInts(x, y), x, y);
      }
    }
  }

  @Override
  default void setElementAt(@NotNull Boolean element, int ix, int iy)
  {
    setValueAt(element, ix, iy);
  }

  @NotNull
  @Override
  default OneDimensionalBooleanAccess subAtX(int ix)
  {
    return new OneDimensionalBooleanAccess()
    {
      @Override
      public void setValueAt(boolean value, int index)
      {
        TwoDimensionalBooleanAccess.this.setValueAt(value, ix, index);
      }

      @Override
      public boolean getValueAt(int index)
      {
        return TwoDimensionalBooleanAccess.this.getValueAt(ix, index);
      }

      @Override
      public int size()
      {
        return sizeY();
      }
    };
  }

  @NotNull
  @Override
  default OneDimensionalBooleanAccess subAtY(int iy)
  {
    return new OneDimensionalBooleanAccess()
    {
      @Override
      public void setValueAt(boolean value, int index)
      {
        TwoDimensionalBooleanAccess.this.setValueAt(value, index, iy);
      }

      @Override
      public boolean getValueAt(int index)
      {
        return TwoDimensionalBooleanAccess.this.getValueAt(index, iy);
      }

      @Override
      public int size()
      {
        return sizeX();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalBooleanAccess transposed()
  {
    return new TwoDimensionalBooleanAccess()
    {
      @Override
      public void setValueAt(boolean value, int ix, int iy)
      {
        TwoDimensionalBooleanAccess.this.setValueAt(value, iy, ix);
      }

      @Override
      public boolean getValueAt(int ix, int iy)
      {
        return TwoDimensionalBooleanAccess.this.getValueAt(iy, ix);
      }

      @Override
      public int sizeX()
      {
        return TwoDimensionalBooleanAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalBooleanAccess.this.sizeX();
      }
    };
  }
}
