// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Indexable;
import de.caff.generics.Types;
import de.caff.generics.function.IntFunction2;

import java.util.Arrays;
import java.util.Collection;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public class TwoDimensionalArray<T>
        extends TwoDimensionalReadAccess.Base<T>
        implements TwoDimensionalAccess<T>
{
  // using arrays here for speed although they don't combine well with generics
  @NotNull
  private final Object[][] elements;

  /**
   * Constructor.
   * This will init all cells with {@code null}.
   * @param sx size in X direction (at least {@code 1})
   * @param sy size in Y direction (at least {@code 1})
   * @see #TwoDimensionalArray(int, int, IntFunction2)
   */
  public TwoDimensionalArray(int sx, int sy)
  {
    if (sx <= 0 || sy <= 0) {
      throw new IllegalArgumentException("Need positive sizes!");
    }
    elements = new Object[sx][sy];
  }

  /**
   * Initializing constructor.
   * @param sx size in X direction (at least {@code 1})
   * @param sy size in Y direction (at least {@code 1})
   * @param filler function called with all index combinations (x,y)
   */
  public TwoDimensionalArray(int sx, int sy, @NotNull IntFunction2<? extends T> filler)
  {
    this(sx, sy);
    fillByIndex(filler);
  }

  /**
   * Constructor.
   * This will call the {@code supplier} for each cell and fill it with the result.
   * @param sx size in X direction (at least {@code 1})
   * @param sy size in Y direction (at least {@code 1})
   * @param supplier supplier for cell content
   */
  public TwoDimensionalArray(@NotNull Supplier<? extends T> supplier, int sx, int sy)
  {
    this(sx, sy);
    for (Object[] row : elements) {
      Types.fillArray(row, supplier);
    }
  }

  /**
   * Constructor.
   * @param sy       size in Y direction (at least {@code 1})
   * @param elements {@code sx * sy} elements
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  public TwoDimensionalArray(int sy,
                             T... elements)
  {
    this(sy, Arrays.asList(elements));
  }

  /**
   * Get the size of the X dimension from the incoming elements size
   * and the Y size.
   * @param ySize        Y size
   * @param completeSize complete size
   * @return X size
   */
  private static int getXSize(int ySize,
                              int completeSize)
  {
    if (completeSize % ySize != 0) {
      throw new IllegalArgumentException("Size of incoming elements has to be a multiple of Y size!");
    }
    return completeSize / ySize;
  }

  /**
   * Constructor.
   * @param sy       size in Y direction (at least {@code 1})
   * @param elements {@code sx * sy} elements
   */
  public TwoDimensionalArray(int sy,
                             @NotNull Collection<T> elements)
  {
    this(getXSize(sy, elements.size()), sy, elements);
  }

  /**
   * Constructor.
   * @param sy       size in Y direction (at least {@code 1})
   * @param elements {@code sx * sy} elements
   */
  public TwoDimensionalArray(int sy,
                             @NotNull Indexable<T> elements)
  {
    this(getXSize(sy, elements.size()), sy, elements);
  }

  /**
   * Constructor.
   *
   * @param sx       size in X direction (at least {@code 1})
   * @param sy       size in Y direction (at least {@code 1})
   * @param elements at least {@code sx * sy} elements
   */
  public TwoDimensionalArray(int sx, int sy,
                             @NotNull Iterable<T> elements)
  {
    this(sx, sy);
    final int size = sx * sy;
    int i = 0;
    for (T elem : elements) {
      if (i == size) {
        return;
      }
      setElementAt(elem, i / sy,  i % sy);
      ++i;
    }
    if (i < size) {
      throw new IllegalArgumentException("Not enough elements in points to fill the ");
    }
  }

  /**
   * Copy constructor.
   * @param source source access
   */
  public TwoDimensionalArray(@NotNull TwoDimensionalReadAccess<T> source)
  {
    elements = new Object[source.sizeX()][source.sizeY()];
    for (int i = source.sizeX() - 1;  i >= 0;  --i) {
      Object[] row = elements[i];
      for (int j = source.sizeY() - 1;  j >= 0;  --j) {
        row[j] = source.getElementAt(i, j);
      }
    }
  }

  @Override
  public void setElementAt(T element, int ix, int iy)
  {
    elements[ix][iy] = element;
  }

  @Override
  public int sizeX()
  {
    return elements.length;
  }

  @Override
  public int sizeY()
  {
    return elements[0].length;
  }

  @Override
  @SuppressWarnings("unchecked") // because setElementAt() only access Ts
  public T getElementAt(int ix, int iy)
  {
    return (T)elements[ix][iy];
  }

  /**
   * Get an independent copy with a possibly different element type.
   * @param copier copier which copies elements from this 2dimensional array to the result array.
   * @param <R> target element type
   * @return 2dimensional array with same size but copied elements
   */
  @NotNull
  public <R> TwoDimensionalArray<R> getCopy(@NotNull Function<? super T, ? extends R> copier)
  {
    return resized(sizeX(), sizeY(), copier, null);
  }

  /**
   * Get a resized copy of this array which contains the same elements for equal indexes.
   * @param sx   new size in X direction
   * @param sy   new size in Y direction
   * @return resized copy of this 2D array where newly created cells are filled with {@code null}
   */
  @NotNull
  public TwoDimensionalArray<T> resized(int sx, int sy)
  {
    return resized(sx, sy, e -> e, null);
  }

  /**
   * Get a resized copy of this array.
   * @param sx   new size in X direction
   * @param sy   new size in Y direction
   * @param filler filler for newly created cells, use {@code null} to leave them empty
   * @return resized copy of this 2D array
   */
  @NotNull
  public TwoDimensionalArray<T> resized(int sx, int sy,
                                       @Nullable IntFunction2<? extends T> filler)
  {
    return resized(sx, sy, e -> e, filler);
  }

  /**
   * Get a resized copy of this array.
   * This will copy elements from this array to the same position in the new array,
   * and fill any elements in the new array whcih have no counterpart in this array.
   * @param sx   new size in X direction
   * @param sy   new size in Y direction
   * @param copier function which copies elements
   * @param filler filler for newly created cells, use {@code null} to leave them empty
   * @return resized independent copy of this 2D array
   * @param <R> possibly different type of resulting array
   */
  @NotNull
  public <R> TwoDimensionalArray<R> resized(int sx, int sy,
                                            @NotNull Function<? super T, ? extends R> copier,
                                            @Nullable IntFunction2<? extends R> filler)
  {
    final int oldSX = sizeX();
    final int oldSY = sizeY();
    final TwoDimensionalArray<R> result = new TwoDimensionalArray<>(sx, sy);
    result.fillByIndex((ix, iy) -> ix < oldSX && iy < oldSY
            ? copier.apply(TwoDimensionalArray.this.getElementAt(ix, iy))
            : (filler != null
                       ? filler.applyAsInt(ix, iy)
                       : null));
    return result;
  }
}
