// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Indexable;
import de.caff.generics.function.*;

import java.util.Collection;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * A two-dimensional rectangular array of items.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface TwoDimensionalAccess<T>
        extends TwoDimensionalReadAccess<T>,
                MultiDimensionalAccess<T>
{
  /**
   * Set the element at the given indexes.
   * @param element element to set
   * @param ix index in X direction from {@code 0} (included) to {@link #sizeX()} (excluded)
   * @param iy index in Y direction from {@code 0} (included) to {@link #sizeY()} (excluded)
   */
  void setElementAt(T element, int ix, int iy);

  @Override
  default void setElement(T value, int... indexes)
  {
    if (indexes.length != 2) {
      throw new IllegalArgumentException("Need 2 indexes for 2-dimensional access!");
    }
    setElementAt(value, indexes[0], indexes[1]);
  }

  /**
   * Get a 1-dimensional view of the row at the given X index.
   * This is a view, so setting elements will change values in this array.
   * @param ix X index
   * @return 1-dimensional array at {@code ix}
   */
  @Override
  @NotNull
  default OneDimensionalAccess<T> subAtX(int ix)
  {
    return new OneDimensionalAccess<T>()
    {
      @Override
      public void set(int index, T elem)
      {
        TwoDimensionalAccess.this.setElementAt(elem, ix, index);
      }

      @Override
      public int size()
      {
        return sizeY();
      }

      @Override
      public T get(int index)
      {
        return TwoDimensionalAccess.this.getElementAt(ix, index);
      }
    };
  }

  /**
   * Get a 1-dimensional view of the column at the given X index.
   * This is a view, so setting elements will change values in this array.
   * @param iy Y index
   * @return 1-dimensional array at {@code iy}
   */
  @Override
  @NotNull
  default OneDimensionalAccess<T> subAtY(int iy)
  {
    return new OneDimensionalAccess<T>()
    {
      @Override
      public void set(int index, T elem)
      {
        TwoDimensionalAccess.this.setElementAt(elem, index, iy);
      }

      @Override
      public int size()
      {
        return sizeX();
      }

      @Override
      public T get(int index)
      {
        return TwoDimensionalAccess.this.getElementAt(index, iy);
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalAccess<T> transposed()
  {
    return new TwoDimensionalAccess<T>()
    {
      @Override
      public void setElementAt(T element, int ix, int iy)
      {
        TwoDimensionalAccess.this.setElementAt(element, iy, ix);
      }

      @Override
      public int sizeX()
      {
        return TwoDimensionalAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return TwoDimensionalAccess.this.sizeX();
      }

      @Override
      public T getElementAt(int ix, int iy)
      {
        return TwoDimensionalAccess.this.getElementAt(iy, ix);
      }
    };
  }

  @Override
  default void changeAll(@NotNull BiFunction<? super T, int[], ? extends T> operator)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        setElementAt(operator.apply(getElementAt(x, y), new int[] { x , y }), x, y);
      }
    }
  }

  @Override
  default void changeAll(@NotNull Function<? super T, ? extends T> operator)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        setElementAt(operator.apply(getElementAt(x, y)), x, y);
      }
    }
  }

  /**
   * Set all values depending on their indexes.
   * This will call the setter for all indexes (x,y) and set the value at the given indexes
   * from its result.
   * @param setter setter which provides the value to set for each index combination
   */
  default void fillByIndex(@NotNull IntFunction2<? extends T> setter)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        setElementAt(setter.applyAsInt(x, y), x, y);
      }
    }
  }

  /**
   * Get an empty 2dimensional access.
   * @param <E> element type
   * @return empty access with size 0 in both X and Y
   */
  @NotNull
  @SuppressWarnings("unchecked")  // because EMPTY holds no elements
  static <E> TwoDimensionalAccess<E> empty()
  {
    return (TwoDimensionalAccess<E>)EMPTY;
  }

  /**
   * Get e 2dimensional access with zero elements in X direction.
   * @param numY number of elements in Y direction
   * @param <E> element type
   * @return access with no size in X
   */
  @NotNull
  static <E> TwoDimensionalAccess<E> zeroX(int numY)
  {
    if (numY == 0) {
      return empty();
    }
    if (numY < 0) {
      throw new IllegalArgumentException("numY has to be non-negative!");
    }
    return new TwoDimensionalAccess<E>()
    {
      @Override
      public void setElementAt(E element, int ix, int iy)
      {
        throw new IndexOutOfBoundsException("No setting for access with no X size!");
      }

      @Override
      public int sizeX()
      {
        return 0;
      }

      @Override
      public int sizeY()
      {
        return numY;
      }

      @Override
      public E getElementAt(int ix, int iy)
      {
        throw new IndexOutOfBoundsException("No getting from access with no X size!");
      }

      @NotNull
      @Override
      public TwoDimensionalAccess<E> transposed()
      {
        return zeroY(numY);
      }

      @Override
      public void changeAll(@NotNull BiFunction<? super E, int[], ? extends E> operator)
      {
      }

      @Override
      public void changeAll(@NotNull Function<? super E, ? extends E> operator)
      {
      }

      @Override
      public MultiDimensionalAccess<E> setAll(@NotNull Function<int[], ? extends E> provider)
      {
        return this;
      }

      @Override
      public long getNumElements()
      {
        return 0L;
      }
    };
  }

  /**
   * Get e 2dimensional access with zero elements in Y direction.
   * @param numX number of elements in X direction
   * @param <E> element type
   * @return access with no size in Y
   */
  @NotNull
  static <E> TwoDimensionalAccess<E> zeroY(int numX)
  {
    if (numX == 0) {
      return empty();
    }
    if (numX < 0) {
      throw new IllegalArgumentException("numX has to be non-negative!");
    }
    return new TwoDimensionalAccess<E>()
    {
      @Override
      public void setElementAt(E element, int ix, int iy)
      {
        throw new IndexOutOfBoundsException("No setting for access with no Y size!");
      }

      @Override
      public int sizeX()
      {
        return numX;
      }

      @Override
      public int sizeY()
      {
        return 0;
      }

      @Override
      public E getElementAt(int ix, int iy)
      {
        throw new IndexOutOfBoundsException("No getting from access with no Y size!");
      }

      @NotNull
      @Override
      public TwoDimensionalAccess<E> transposed()
      {
        return zeroX(numX);
      }

      @Override
      public void changeAll(@NotNull BiFunction<? super E, int[], ? extends E> operator)
      {
      }

      @Override
      public void changeAll(@NotNull Function<? super E, ? extends E> operator)
      {
      }

      @Override
      public MultiDimensionalAccess<E> setAll(@NotNull Function<int[], ? extends E> provider)
      {
        return this;
      }

      @Override
      public long getNumElements()
      {
        return 0L;
      }
    };
  }

  /**
   * Create a 2dimensional array with the given sizes
   * with elements initialized to null.
   * @param sizeX size in X direction
   * @param sizeY size in Y direction
   * @param <E> element type
   * @return 2dimensional array with the given size
   */
  @NotNull
  static <E> TwoDimensionalAccess<E> createNulled(int sizeX, int sizeY)
  {
    if (sizeX == 0) {
      return zeroX(sizeY);
    }
    if (sizeY == 0) {
      return zeroY(sizeX);
    }
    return new TwoDimensionalArray<>(sizeX, sizeY);
  }

  /**
   * Create a 2dimensional array with the given sizes
   * with elements initialized to a given constant.
   * @param constant constant to which all elements will be initialized
   * @param sizeX size in X direction
   * @param sizeY size in Y direction
   * @param <E> element type
   * @return 2dimensional array with the given size
   */
  @NotNull
  static <E> TwoDimensionalAccess<E> createConstant(@Nullable E constant,
                                                    int sizeX, int sizeY)
  {
    return constant == null
            ? createNulled(sizeX, sizeY)
            : createSupplied(() -> constant, sizeX, sizeY);
  }

  /**
   * Create a 2dimensional array with the given sizes
   * with elements initialized by a supplier.
   * @param supplier supplier for the initial elements
   * @param sizeX size in X direction
   * @param sizeY size in Y direction
   * @param <E> element type
   * @return 2dimensional array with the given size
   */
  @NotNull
  static <E> TwoDimensionalAccess<E> createSupplied(@NotNull Supplier<? extends E> supplier,
                                                    int sizeX, int sizeY)
  {
    if (sizeX == 0) {
      return zeroX(sizeY);
    }
    if (sizeY == 0) {
      return zeroY(sizeX);
    }
    return new TwoDimensionalArray<>(supplier, sizeX, sizeY);
  }

  /**
   * Create a 2dimensional array with the given sizes
   * with elements initialized by their indices.
   * @param supplier supplier for the initial elements, gets both X and Y index as arguments
   * @param sizeX size in X direction
   * @param sizeY size in Y direction
   * @param <E> element type
   * @return 2dimensional array with the given size
   */
  @NotNull
  static <E> TwoDimensionalAccess<E> createIndexed(@NotNull BiFunction<Integer, Integer, ? extends E> supplier,
                                                   int sizeX, int sizeY)
  {
    if (sizeX == 0) {
      return zeroX(sizeY);
    }
    if (sizeY == 0) {
      return zeroY(sizeX);
    }
    final TwoDimensionalAccess<E> array = new TwoDimensionalArray<>(sizeX, sizeY);
    for (int x = 0;  x < sizeX;  ++x) {
      for (int y = 0;  y < sizeY;  ++y) {
        array.setElementAt(supplier.apply(x, y), x, y);
      }
    }
    return array;
  }

  /**
   * Empty 2dim array.
   * Use {@link #empty()} instead.
   */
  TwoDimensionalAccess<Object> EMPTY = new TwoDimensionalAccess<Object>()
  {
    @Override
    public void setElementAt(Object element, int ix, int iy)
    {
      throw new IndexOutOfBoundsException("Cannot set elements of empty 2dimensional array!");
    }

    @Override
    public int sizeX()
    {
      return 0;
    }

    @Override
    public int sizeY()
    {
      return 0;
    }

    @Override
    public Object getElementAt(int ix, int iy)
    {
      throw new IndexOutOfBoundsException("Cannot get elements from empty 2dimensional array!");
    }

    @Override
    public void setElement(Object value, int... indexes)
    {
      throw new IndexOutOfBoundsException("Cannot get elements from empty 2dimensional array!");
    }

    @NotNull
    @Override
    public OneDimensionalAccess<Object> subAtX(int ix)
    {
      throw new IndexOutOfBoundsException("Cannot get sub array from empty 2dimensional array!");
    }

    @NotNull
    @Override
    public OneDimensionalAccess<Object> subAtY(int iy)
    {
      throw new IndexOutOfBoundsException("Cannot get sub array from empty 2dimensional array!");
    }

    @NotNull
    @Override
    public TwoDimensionalAccess<Object> transposed()
    {
      return this;
    }

    @Override
    public void changeAll(@NotNull BiFunction<? super Object, int[], ?> operator)
    {
    }

    @Override
    public void changeAll(@NotNull Function<? super Object, ?> operator)
    {
    }

    @Override
    public Object change(@NotNull Function<? super Object, ?> operator, int... indexes)
    {
      throw new IndexOutOfBoundsException("Cannot change elements of empty 2dimensional array!");
    }

    @Override
    public void setFrom(@NotNull MultiIndexLinearizer.Sequencer sequencer, @NotNull Collection<?> elements)
    {
      if (!elements.isEmpty()) {
        throw new IndexOutOfBoundsException("Cannot set elemeets of empty 2dimensional array!");
      }
    }

    @Override
    public void setFrom(@NotNull MultiIndexLinearizer.Sequencer sequencer, @NotNull Indexable<?> elements)
    {
      if (!elements.isEmpty()) {
        throw new IndexOutOfBoundsException("Cannot set elemeets of empty 2dimensional array!");
      }
    }

    @Override
    public void setFrom(@NotNull MultiIndexLinearizer.Sequencer sequencer, @NotNull Iterable<?> elements)
    {
      if (elements.iterator().hasNext()) {
        throw new IndexOutOfBoundsException("Cannot set elemeets of empty 2dimensional array!");
      }
    }

    @Override
    public MultiDimensionalAccess<Object> setAll(@NotNull Function<int[], ?> provider)
    {
      return this;
    }

    @Override
    public Object getElement(int... indexes)
    {
      throw new IndexOutOfBoundsException("Cannot get elements from empty 2dimensional array!");
    }

    @Override
    public int getNumDimensions()
    {
      return 2;
    }

    @Override
    public int getSize(int dim)
    {
      return 0;
    }

    @NotNull
    @Override
    public int[] getSizes()
    {
      return new int[] {0, 0};
    }

    @Override
    public long getNumElements()
    {
      return 0L;
    }

    @NotNull
    @Override
    public <TOUT> TwoDimensionalReadAccess<TOUT> view(@NotNull Function<Object, TOUT> conv)
    {
      return empty();
    }

    @Override
    public void visitAll(@NotNull Procedure2<? super Object, int[]> visitor)
    {
    }

    @Override
    public void visitAll(@NotNull Procedure1<? super Object> visitor)
    {
    }
  };

}
