// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.IntToShortFunction3;
import de.caff.generics.function.ShortOperator1;

/**
 * Read/write access for 3-dimensional rectangular short arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface ThreeDimensionalShortAccess
        extends ThreeDimensionalShortReadAccess,
                ThreeDimensionalAccess<Short>
{
  /** Empty 3dimensional short access. */
  ThreeDimensionalShortAccess EMPTY = new ThreeDimensionalShortAccess()
  {
    @Override
    public void setValueAt(short value, int ix, int iy, int iz)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public short getValueAt(int ix, int iy, int iz)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int sizeX()
    {
      return 0;
    }

    @Override
    public int sizeY()
    {
      return 0;
    }

    @Override
    public int sizeZ()
    {
      return 0;
    }
  };

  /**
   * Set the value at the given indexes.
   * @param value  value to set
   * @param ix     X index
   * @param iy     Y index
   * @param iz     Z index
   */
  void setValueAt(short value, int ix, int iy, int iz);

  /**
   * Change the value at the given indexes.
   * @param operator operator to apply to the value
   * @param ix       X index
   * @param iy       Y index
   * @param iz       Z index
   */
  default void changeValueAt(@NotNull ShortOperator1 operator, int ix, int iy, int iz)
  {
    setValueAt(operator.applyAsShort(getValueAt(ix, iy, iz)), ix, iy, iz);
  }

  /**
   * Change all values by applying an operator.
   * @param operator operator to apply
   */
  default void changeAllValues(@NotNull ShortOperator1 operator)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    final int sz = sizeZ();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        for (int z = 0;  z < sz;  ++z) {
          changeValueAt(operator, x, y, z);
        }
      }
    }
  }

  /**
   * Set all values depending on their indexes.
   * This will call the setter for all indexes (x,y,z) and set the value at the given indexes
   * from its result.
   * @param setter setter which provides the value to set for each index combination
   */
  default void fillValuesByIndex(@NotNull IntToShortFunction3 setter)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    final int sz = sizeZ();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        for (int z = 0;  z < sz;  ++z) {
          setElementAt(setter.applyAsShort(x, y, z), x, y, z);
        }
      }
    }
  }

  @Override
  default void setElementAt(@NotNull Short element, int ix, int iy, int iz)
  {
    setValueAt(element, ix, iy, iz);
  }

  @NotNull
  @Override
  default TwoDimensionalShortAccess subAtX(int fixX)
  {
    return new TwoDimensionalShortAccess()
    {
      @Override
      public void setValueAt(short value, int ix, int iy)
      {
        ThreeDimensionalShortAccess.this.setValueAt(value, fixX, ix, iy);
      }

      @Override
      public short getValueAt(int ix, int iy)
      {
        return ThreeDimensionalShortAccess.this.getValueAt(fixX, ix, iy);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalShortAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalShortAccess.this.sizeZ();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalShortAccess subAtY(int fixY)
  {
    return new TwoDimensionalShortAccess()
    {
      @Override
      public void setValueAt(short value, int ix, int iy)
      {
        ThreeDimensionalShortAccess.this.setValueAt(value, ix, fixY, iy);
      }

      @Override
      public short getValueAt(int ix, int iy)
      {
        return ThreeDimensionalShortAccess.this.getValueAt(ix, fixY, iy);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalShortAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalShortAccess.this.sizeZ();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalShortAccess subAtZ(int fixZ)
  {
    return new TwoDimensionalShortAccess()
    {
      @Override
      public void setValueAt(short value, int ix, int iy)
      {
        ThreeDimensionalShortAccess.this.setValueAt(value, ix, iy, fixZ);
      }

      @Override
      public short getValueAt(int ix, int iy)
      {
        return ThreeDimensionalShortAccess.this.getValueAt(ix, iy, fixZ);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalShortAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalShortAccess.this.sizeY();
      }
    };
  }
}
