// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Function1;
import de.caff.generics.function.Procedure1;
import de.caff.generics.function.Procedure2;

/**
 * Read access for 3-dimensional rectangular arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface ThreeDimensionalReadAccess<T>
        extends MultiDimensionalReadAccess<T>
{
  /** Empty 3dimensional read access. Use {@link #empty()} instead. */
  ThreeDimensionalReadAccess<?> EMPTY = new ThreeDimensionalReadAccess<Object>()
  {
    @Override
    public int sizeX()
    {
      return 0;
    }

    @Override
    public int sizeY()
    {
      return 0;
    }

    @Override
    public int sizeZ()
    {
      return 0;
    }

    @Override
    public Object getElementAt(int ix, int iy, int iz)
    {
      throw new IndexOutOfBoundsException();
    }
  };

  /**
   * Get the number of elements in X direction.
   * @return number of elements in X direction
   */
  int sizeX();

  /**
   * Get the number of elements in Y direction
   * @return number of elements in Y direction
   */
  int sizeY();

  /**
   * Get the number of elements in Z direction
   * @return number of elements in Z direction
   */
  int sizeZ();

  /**
   * Get the element at the given indexes.
   * @param ix index in X direction from {@code 0} (included) to {@link #sizeX()} (excluded)
   * @param iy index in Y direction from {@code 0} (included) to {@link #sizeY()} (excluded)
   * @param iz index in Z direction from {@code 0} (included) to {@link #sizeZ()} (excluded)
   * @return element at the given indexes
   */
  T getElementAt(int ix, int iy, int iz);

  @Override
  default T getElement(int... indexes)
  {
    if (indexes.length != 3) {
      throw new IllegalArgumentException("Need 3 indexes for 3-dimensional array!");
    }
    return getElementAt(indexes[0], indexes[1], indexes[2]);
  }

  /**
   * Get a 2-dimensional view of the plane at the given X index.
   * This is a view, so setting elements will change values in this array.
   * @param fixX fix X index
   * @return 2-dimensional array at {@code fixX}
   */
  @NotNull
  default TwoDimensionalReadAccess<T> subAtX(int fixX)
  {
    return new TwoDimensionalReadAccess<T>()
    {
      @Override
      public int sizeX()
      {
        return ThreeDimensionalReadAccess.this.sizeY();
      }

      public int sizeY() { return ThreeDimensionalReadAccess.this.sizeZ(); }

      @Override
      public T getElementAt(int ix, int iy)
      {
        return ThreeDimensionalReadAccess.this.getElementAt(fixX, ix, iy);
      }
    };
  }

  /**
   * Get a 2-dimensional view of the plane at the given Y index.
   * This is a view, so setting elements will change values in this array.
   * @param fixY fix Y index
   * @return 2-dimensional array at {@code fixY}
   */
  @NotNull
  default TwoDimensionalReadAccess<T> subAtY(int fixY)
  {
    return new TwoDimensionalReadAccess<T>()
    {
      @Override
      public int sizeX()
      {
        return ThreeDimensionalReadAccess.this.sizeX();
      }

      public int sizeY() { return ThreeDimensionalReadAccess.this.sizeZ(); }

      @Override
      public T getElementAt(int ix, int iy)
      {
        return ThreeDimensionalReadAccess.this.getElementAt(ix, fixY, iy);
      }
    };
  }

  /**
   * Get a 2-dimensional view of the plane at the given Z index.
   * This is a view, so setting elements will change values in this array.
   * @param fixZ fix Z index
   * @return 2-dimensional array at {@code fixZ}
   */
  @NotNull
  default TwoDimensionalReadAccess<T> subAtZ(int fixZ)
  {
    return new TwoDimensionalReadAccess<T>()
    {
      @Override
      public int sizeX()
      {
        return ThreeDimensionalReadAccess.this.sizeX();
      }

      public int sizeY() { return ThreeDimensionalReadAccess.this.sizeY(); }

      @Override
      public T getElementAt(int ix, int iy)
      {
        return ThreeDimensionalReadAccess.this.getElementAt(ix, iy, fixZ);
      }
    };
  }

  @Override
  default int getNumDimensions()
  {
    return 3;
  }

  @Override
  default int getSize(int dim)
  {
    switch (dim) {
    case 0:
      return sizeX();
    case 1:
      return sizeY();
    case 2:
      return sizeZ();
    }
    throw new IllegalArgumentException("There are only 3 dimensions!");
  }

  @NotNull
  @Override
  default int[] getSizes()
  {
    return new int[] { sizeX(), sizeY(), sizeZ() };
  }

  @Override
  default long getNumElements()
  {
    return (long)sizeX() * sizeY() * sizeZ();
  }

  /**
   * Get a view of this two dimensional array with converted elements.
   * @param conv   element converter
   * @param <TOUT>  element type provided by the view
   * @return view of this array with converted elements
   */
  @NotNull
  default <TOUT> ThreeDimensionalReadAccess<TOUT> view(@NotNull Function1<TOUT, T> conv)
  {
    return new ThreeDimensionalReadAccess<TOUT>()
    {
      @Override
      public int sizeX()
      {
        return ThreeDimensionalReadAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalReadAccess.this.sizeY();
      }

      @Override
      public int sizeZ()
      {
        return ThreeDimensionalReadAccess.this.sizeZ();
      }

      @Override
      public TOUT getElementAt(int ix, int iy, int iz)
      {
        return conv.apply(ThreeDimensionalReadAccess.this.getElementAt(ix, iy, iz));
      }
    };
  }

  @Override
  default void visitAll(@NotNull Procedure2<? super T, int[]> visitor)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    final int sz = sizeZ();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        for (int z = 0;  z < sz;  ++z) {
          visitor.apply(getElementAt(x, y, z), new int[] { x, y, z });
        }
      }
    }
  }

  @Override
  default void visitAll(@NotNull Procedure1<? super T> visitor)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    final int sz = sizeZ();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        for (int z = 0;  z < sz;  ++z) {
          visitor.apply(getElementAt(x, y, z));
        }
      }
    }
  }

  /**
   * Get an empty 3dimensional read access.
   * @param <E> element type
   * @return {@link #EMPTY} in a form the compiler accepts
   */
  @NotNull
  @SuppressWarnings("unchecked") // as EMPTY is immutable
  static <E> ThreeDimensionalReadAccess<E> empty()
  {
    return (ThreeDimensionalReadAccess<E>)EMPTY;
  }

  /**
   * Get a 3dimensional read access for a single element.
   * @param singleElement single element
   * @return 2dimensional read access with size 1 in all 3 directions,
   *         containing just the given single element
   * @param <E> element type
   */
  @NotNull
  static <E> ThreeDimensionalReadAccess<E> singleton(E singleElement)
  {
    return new ThreeDimensionalReadAccess<E>()
    {
      @Override
      public int sizeX()
      {
        return 1;
      }

      @Override
      public int sizeY()
      {
        return 1;
      }

      @Override
      public int sizeZ()
      {
        return 1;
      }

      @Override
      public E getElementAt(int ix, int iy, int iz)
      {
        if (ix != 0  ||  iy != 0  ||  iz != 0) {
          throw new IndexOutOfBoundsException();
        }
        return singleElement;
      }
    };
  }
}
