// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;

/**
 * Read access for 3-dimensional rectangular long arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface ThreeDimensionalLongReadAccess
        extends ThreeDimensionalReadAccess<Long>
{
  /**
   * Get the value at the given indexes.
   * @param ix X index
   * @param iy Y index
   * @param iz Z index
   * @return value at the given index
   */
  long getValueAt(int ix, int iy, int iz);

  @Override
  @NotNull
  default Long getElementAt(int ix, int iy, int iz)
  {
    return getValueAt(ix, iy, iz);
  }

  @NotNull
  @Override
  default TwoDimensionalLongReadAccess subAtX(int fixX)
  {
    return new TwoDimensionalLongReadAccess()
    {
      @Override
      public long getValueAt(int ix, int iy)
      {
        return ThreeDimensionalLongReadAccess.this.getValueAt(fixX, ix, iy);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalLongReadAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalLongReadAccess.this.sizeZ();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalLongReadAccess subAtY(int fixY)
  {
    return new TwoDimensionalLongReadAccess()
    {
      @Override
      public long getValueAt(int ix, int iy)
      {
        return ThreeDimensionalLongReadAccess.this.getValueAt(ix, fixY, iy);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalLongReadAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalLongReadAccess.this.sizeZ();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalLongReadAccess subAtZ(int fixZ)
  {
    return new TwoDimensionalLongReadAccess()
    {
      @Override
      public long getValueAt(int ix, int iy)
      {
        return ThreeDimensionalLongReadAccess.this.getValueAt(ix, iy, fixZ);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalLongReadAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalLongReadAccess.this.sizeY();
      }
    };
  }

}
