// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;

/**
 * Read access for 3-dimensional rectangular int arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface ThreeDimensionalIntReadAccess
        extends ThreeDimensionalReadAccess<Integer>
{
  /** Empty 3dimensional int read access. */
  ThreeDimensionalIntReadAccess EMPTY = new ThreeDimensionalIntReadAccess()
  {
    @Override
    public int getValueAt(int ix, int iy, int iz)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int sizeX()
    {
      return 0;
    }

    @Override
    public int sizeY()
    {
      return 0;
    }

    @Override
    public int sizeZ()
    {
      return 0;
    }
  };

  /**
   * Get a 3dimensional int read access for a single element.
   * @param singleElement single element
   * @return 3dimensional int read access with size 1 in all 3 directions,
   *         containing just the given single element
   */
  @NotNull
  static ThreeDimensionalIntReadAccess singleton(int singleElement)
  {
    return new ThreeDimensionalIntReadAccess()
    {
      @Override
      public int getValueAt(int ix, int iy, int iz)
      {
        if (ix != 0  ||  iy != 0  ||  iz != 0) {
          throw new IllegalArgumentException();
        }
        return singleElement;
      }

      @Override
      public int sizeX()
      {
        return 1;
      }

      @Override
      public int sizeY()
      {
        return 1;
      }

      @Override
      public int sizeZ()
      {
        return 1;
      }
    };
  }

  /**
   * Get the value at the given indexes.
   * @param ix X index
   * @param iy Y index
   * @param iz Z index
   * @return value at the given index
   */
  int getValueAt(int ix, int iy, int iz);

  @Override
  @NotNull
  default Integer getElementAt(int ix, int iy, int iz)
  {
    return getValueAt(ix, iy, iz);
  }

  @NotNull
  @Override
  default TwoDimensionalIntReadAccess subAtX(int fixX)
  {
    return new TwoDimensionalIntReadAccess()
    {
      @Override
      public int getValueAt(int ix, int iy)
      {
        return ThreeDimensionalIntReadAccess.this.getValueAt(fixX, ix, iy);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalIntReadAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalIntReadAccess.this.sizeZ();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalIntReadAccess subAtY(int fixY)
  {
    return new TwoDimensionalIntReadAccess()
    {
      @Override
      public int getValueAt(int ix, int iy)
      {
        return ThreeDimensionalIntReadAccess.this.getValueAt(ix, fixY, iy);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalIntReadAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalIntReadAccess.this.sizeZ();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalIntReadAccess subAtZ(int fixZ)
  {
    return new TwoDimensionalIntReadAccess()
    {
      @Override
      public int getValueAt(int ix, int iy)
      {
        return ThreeDimensionalIntReadAccess.this.getValueAt(ix, iy, fixZ);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalIntReadAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalIntReadAccess.this.sizeY();
      }
    };
  }

}
