// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.IntToDoubleFunction3;

import java.util.function.DoubleUnaryOperator;

/**
 * Read/write access for 3-dimensional rectangular double arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface ThreeDimensionalDoubleAccess
        extends ThreeDimensionalDoubleReadAccess,
                ThreeDimensionalAccess<Double>
{
  /** Empty 3dimensional double access. */
  ThreeDimensionalDoubleAccess EMPTY = new ThreeDimensionalDoubleAccess()
  {
    @Override
    public void setValueAt(double value, int ix, int iy, int iz)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public double getValueAt(int ix, int iy, int iz)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int sizeX()
    {
      return 0;
    }

    @Override
    public int sizeY()
    {
      return 0;
    }

    @Override
    public int sizeZ()
    {
      return 0;
    }
  };

  /**
   * Set the value at the given indexes.
   * @param value  value to set
   * @param ix     X index
   * @param iy     Y index
   * @param iz     Z index
   */
  void setValueAt(double value, int ix, int iy, int iz);

  /**
   * Change the value at the given indexes.
   * @param operator operator to apply to the value
   * @param ix       X index
   * @param iy       Y index
   * @param iz       Z index
   */
  default void changeValueAt(@NotNull DoubleUnaryOperator operator, int ix, int iy, int iz)
  {
    setValueAt(operator.applyAsDouble(getValueAt(ix, iy, iz)), ix, iy, iz);
  }

  /**
   * Change all values by applying an operator.
   * @param operator operator to apply
   */
  default void changeAllValues(@NotNull DoubleUnaryOperator operator)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    final int sz = sizeZ();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        for (int z = 0;  z < sz;  ++z) {
          changeValueAt(operator, x, y, z);
        }
      }
    }
  }

  /**
   * Set all values depending on their indexes.
   * This will call the setter for all indexes (x,y,z) and set the value at the given indexes
   * from its result.
   * @param setter setter which provides the value to set for each index combination
   */
  default void fillValuesByIndex(@NotNull IntToDoubleFunction3 setter)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    final int sz = sizeZ();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        for (int z = 0;  z < sz;  ++z) {
          setElementAt(setter.applyAsDouble(x, y, z), x, y, z);
        }
      }
    }
  }

  @Override
  default void setElementAt(@NotNull Double element, int ix, int iy, int iz)
  {
    setValueAt(element, ix, iy, iz);
  }

  @NotNull
  @Override
  default TwoDimensionalDoubleAccess subAtX(int fixX)
  {
    return new TwoDimensionalDoubleAccess()
    {
      @Override
      public void setValueAt(double value, int ix, int iy)
      {
        ThreeDimensionalDoubleAccess.this.setValueAt(value, fixX, ix, iy);
      }

      @Override
      public double getValueAt(int ix, int iy)
      {
        return ThreeDimensionalDoubleAccess.this.getValueAt(fixX, ix, iy);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalDoubleAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalDoubleAccess.this.sizeZ();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalDoubleAccess subAtY(int fixY)
  {
    return new TwoDimensionalDoubleAccess()
    {
      @Override
      public void setValueAt(double value, int ix, int iy)
      {
        ThreeDimensionalDoubleAccess.this.setValueAt(value, ix, fixY, iy);
      }

      @Override
      public double getValueAt(int ix, int iy)
      {
        return ThreeDimensionalDoubleAccess.this.getValueAt(ix, fixY, iy);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalDoubleAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalDoubleAccess.this.sizeZ();
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalDoubleAccess subAtZ(int fixZ)
  {
    return new TwoDimensionalDoubleAccess()
    {
      @Override
      public void setValueAt(double value, int ix, int iy)
      {
        ThreeDimensionalDoubleAccess.this.setValueAt(value, ix, iy, fixZ);
      }

      @Override
      public double getValueAt(int ix, int iy)
      {
        return ThreeDimensionalDoubleAccess.this.getValueAt(ix, iy, fixZ);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalDoubleAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalDoubleAccess.this.sizeY();
      }
    };
  }
}
