// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.IntToByteFunction3;

/**
 * A 3-dimensional array of byte values.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public class ThreeDimensionalByteArray
        implements ThreeDimensionalByteAccess
{
  @NotNull
  private final byte[][][] values;

  /**
   * Constructor.
   * This will fill all values with {@code 0}.
   * @param xSize X size of array
   * @param ySize Y size of array
   * @param zSize Z size of array
   * @see #ThreeDimensionalByteArray(int, int, int, IntToByteFunction3)
   */
  public ThreeDimensionalByteArray(int xSize, int ySize, int zSize)
  {
    values = new byte[xSize][ySize][zSize];
  }

  /**
   * Initializing constructor.
   * @param xSize X size of array
   * @param ySize Y size of array
   * @param zSize Z size of array
   * @param filler setter function which provides the value to set for each index combination
   * @see #fillValuesByIndex(de.caff.generics.function.IntToByteFunction3)
   */
  public ThreeDimensionalByteArray(int xSize, int ySize, int zSize, @NotNull IntToByteFunction3 filler)
  {
    this(xSize, ySize, zSize);
    fillValuesByIndex(filler);
  }

  /**
   * Constructor.
   * @param ySize  size in y direction
   * @param zSize  size in z direction
   * @param values values in order, the first {@code zSize} values
   *               will become row 0 of plane 0, the second {@code zSize} values
   *               row 1, and so on until the first plane of {@code ySize} rows is filled.
   *               Then the next plane is filled.
   *               Needs a multiple of {@code ySize*zSize} values, and the multiplicator
   *               defines the X size.
   */
  public ThreeDimensionalByteArray(int ySize, int zSize,
                                   byte... values)
  {
    if (ySize <= 0  ||  values.length == 0) {
      throw new IllegalArgumentException("Need positive sizes!");
    }
    if (values.length % zSize != 0  ||  values.length % ySize != 0) {
      throw new IllegalArgumentException("Need a multiple of ySize*zSize values!");
    }
    final int yzSize = ySize * zSize;
    final int xSize = values.length / yzSize;
    this.values = new byte[xSize][ySize][zSize];
    for (int v = 0;  v < values.length;  ++v) {
      final int x = v / yzSize;
      final int y = (v % yzSize) / zSize;
      final int z = v % zSize;
      this.values[x][y][z] = values[z];
    }
  }

  @Override
  public void setValueAt(byte value, int ix, int iy, int iz)
  {
    values[ix][iy][iz] = value;
  }

  @Override
  public byte getValueAt(int ix, int iy, int iz)
  {
    return values[ix][iy][iz];
  }

  @Override
  public int sizeX()
  {
    return values.length;
  }

  @Override
  public int sizeY()
  {
    return values[0].length;
  }

  @Override
  public int sizeZ()
  {
    return values[0][0].length;
  }
}
