// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;

/**
 * A 4-dimensional array.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public class ThreeDimensionalArray<T>
        implements ThreeDimensionalAccess<T>
{
  // using arrays here for speed although they don't combine well with generics
  @NotNull
  private final Object[][][] elements;

  /**
   * Constructor.
   * @param sx size in X direction (at least {@code 1})
   * @param sy size in Y direction (at least {@code 1})
   * @param sz size in Z direction (at least {@code 1})
   */
  public ThreeDimensionalArray(int sx, int sy, int sz)
  {
    if (sx <= 0 || sy <= 0) {
      throw new IllegalArgumentException("Need positive sizes!");
    }
    elements = new Object[sx][sy][sz];
  }

  /**
   * Copy constructor.
   * @param source source access
   */
  public ThreeDimensionalArray(@NotNull ThreeDimensionalAccess<T> source)
  {
    elements = new Object[source.sizeX()][source.sizeY()][source.sizeZ()];
    for (int i = source.sizeX() - 1;  i >= 0;  --i) {
      final Object[][] plane = elements[i];
      for (int j = source.sizeY() - 1;  j >= 0;  --j) {
        final Object[] row = plane[j];
        for (int k = source.sizeZ() - 1;  k >= 0;  --k) {
          row[k] = source.getElementAt(i, j, k);
        }
      }
    }
  }

  @Override
  public void setElementAt(T element, int ix, int iy, int iz)
  {
    elements[ix][iy][iz] = element;
  }

  @Override
  public int sizeX()
  {
    return elements.length;
  }

  @Override
  public int sizeY()
  {
    return elements[0].length;
  }

  public int sizeZ()
  {
    return elements[0][0].length;
  }

  @Override
  @SuppressWarnings("unchecked") // because setElementAt() only access Ts
  public T getElementAt(int ix, int iy, int iz)
  {
    return (T)elements[ix][iy];
  }
}
