// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.IntFunction3;

import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * A 3-dimensional rectangular array of items.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface ThreeDimensionalAccess<T>
        extends ThreeDimensionalReadAccess<T>,
                MultiDimensionalAccess<T>
{
  /** Empty 3dimensional access. Use {@link #empty()} instead. */
  ThreeDimensionalAccess<?> EMPTY = new ThreeDimensionalAccess<Object>()
  {
    @Override
    public void setElementAt(Object element, int ix, int iy, int iz)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int sizeX()
    {
      return 0;
    }

    @Override
    public int sizeY()
    {
      return 0;
    }

    @Override
    public int sizeZ()
    {
      return 0;
    }

    @Override
    public Object getElementAt(int ix, int iy, int iz)
    {
      throw new IndexOutOfBoundsException();
    }
  };

  /**
   * Set the element at the given indexes.
   * @param element element to set
   * @param ix index in X direction from {@code 0} (included) to {@link #sizeX()} (excluded)
   * @param iy index in Y direction from {@code 0} (included) to {@link #sizeY()} (excluded)
   * @param iz index in Y direction from {@code 0} (included) to {@link #sizeZ()} (excluded)
   */
  void setElementAt(T element, int ix, int iy, int iz);

  @Override
  default void setElement(T value, int... indexes)
  {
    if (indexes.length != 3) {
      throw new IllegalArgumentException("Need 2 indexes for 2-dimensional access!");
    }
    setElementAt(value, indexes[0], indexes[1], indexes[2]);
  }

  @NotNull
  @Override
  default TwoDimensionalAccess<T> subAtX(int fixX)
  {
    return new TwoDimensionalAccess<T>()
    {
      @Override
      public void setElementAt(T element, int ix, int iy)
      {
        ThreeDimensionalAccess.this.setElementAt(element, fixX, ix, iy);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalAccess.this.sizeY();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalAccess.this.sizeZ();
      }

      @Override
      public T getElementAt(int ix, int iy)
      {
        return ThreeDimensionalAccess.this.getElementAt(fixX, ix, iy);
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalReadAccess<T> subAtY(int fixY)
  {
    return new TwoDimensionalAccess<T>()
    {
      @Override
      public void setElementAt(T element, int ix, int iy)
      {
        ThreeDimensionalAccess.this.setElementAt(element, ix, fixY, iy);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalAccess.this.sizeZ();
      }

      @Override
      public T getElementAt(int ix, int iy)
      {
        return ThreeDimensionalAccess.this.getElementAt(ix, fixY, iy);
      }
    };
  }

  @NotNull
  @Override
  default TwoDimensionalReadAccess<T> subAtZ(int fixZ)
  {
    return new TwoDimensionalAccess<T>()
    {
      @Override
      public void setElementAt(T element, int ix, int iy)
      {
        ThreeDimensionalAccess.this.setElementAt(element, ix, iy, fixZ);
      }

      @Override
      public int sizeX()
      {
        return ThreeDimensionalAccess.this.sizeX();
      }

      @Override
      public int sizeY()
      {
        return ThreeDimensionalAccess.this.sizeY();
      }

      @Override
      public T getElementAt(int ix, int iy)
      {
        return ThreeDimensionalAccess.this.getElementAt(ix, iy, fixZ);
      }
    };
  }

  @Override
  default void changeAll(@NotNull BiFunction<? super T, int[], ? extends T> operator)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    final int sz = sizeZ();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        for (int z = 0;  z < sz;  ++z) {
          setElementAt(operator.apply(getElementAt(x, y, z), new int[] { x, y, z }), x, y, z);
        }
      }
    }
  }

  @Override
  default void changeAll(@NotNull Function<? super T, ? extends T> operator)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    final int sz = sizeZ();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        for (int z = 0;  z < sz;  ++z) {
          setElementAt(operator.apply(getElementAt(x, y, z)), x, y, z);
        }
      }
    }
  }

  /**
   * Set all values depending on their indexes.
   * This will call the setter for all indexes (x,y,z) and set the value at the given indexes
   * from its result.
   * @param setter setter which provides the value to set for each index combination
   */
  default void fillByIndex(@NotNull IntFunction3<? extends T> setter)
  {
    final int sx = sizeX();
    final int sy = sizeY();
    final int sz = sizeZ();
    for (int x = 0;  x < sx;  ++x) {
      for (int y = 0;  y < sy;  ++y) {
        for (int z = 0;  z < sz;  ++z) {
          setElementAt(setter.applyAsInt(x, y, z), x, y, z);
        }
      }
    }
  }

  /**
   * Get an empty 3dimensional access.
   * @param <E> element type
   * @return {@link #EMPTY} in a form the compiler accepts
   */
  @NotNull
  @SuppressWarnings("unchecked") // as EMPTY is immutable
  static <E> ThreeDimensionalReadAccess<E> empty()
  {
    return (ThreeDimensionalReadAccess<E>)EMPTY;
  }
}
