// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.IntToCharFunction2;

/**
 * Two-dimensional array of char values optimized for small sizes.
 * <p>
 * This internally maps to a one-dimensional char array which is created faster.
 * Because of the mapping the size of the dimensions is restricted in a way that the
 * number of elements in this array (i.e. the multiplication of its X and Y size) is less
 * than {@link Integer#MAX_VALUE}. Use {@link TwoDimensionalCharArray} if you are not
 * sure to always obey this restriction.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public class SmallTwoDimensionalCharArray
        extends TwoDimensionalCharReadAccess.Base
        implements TwoDimensionalCharAccess
{
  @NotNull
  private final char[] values;
  private final int xSize;
  private final int ySize;

  /**
   * Constructor.
   * This fills all elements with {@code 0.0} values.
   * @param xSize X size of array
   * @param ySize Y size of array
   * @see #SmallTwoDimensionalCharArray(int, int, IntToCharFunction2)
   * @throws IllegalArgumentException on negative sizes or if linearized size becoms too large
   */
  public SmallTwoDimensionalCharArray(int xSize, int ySize)
  {
    if (xSize < 0  ||  ySize < 0) {
      throw new IllegalArgumentException("Sizes have to be non-negative!");
    }
    if (Integer.MAX_VALUE / xSize < ySize) {
      throw new IllegalArgumentException("Sizes overflow the range of integer. Use TwoDimensionalCharArray instead!");
    }
    values = new char[xSize*ySize];
    this.xSize = xSize;
    this.ySize = ySize;
  }

  /**
   * Initializing constructor.
   * @param xSize X size of array
   * @param ySize Y size of array
   * @param filler setter function which provides the value to set for each index combination
   * @see #fillValuesByIndex(IntToCharFunction2)
   * @throws IllegalArgumentException on negative sizes or if linearized size becoms too large
   */
  public SmallTwoDimensionalCharArray(int xSize, int ySize, @NotNull IntToCharFunction2 filler)
  {
    this(xSize, ySize);
    fillValuesByIndex(filler);
  }

  /**
   * Constructor.
   * @param ySize  size in y direction
   * @param values values in order, the first {@code ySize} values
   *               will become row 0, the second {@code ySize} values
   *               row 1, and so on. Needs a multiple of ySize values.
   */
  public SmallTwoDimensionalCharArray(int ySize,
                                      char... values)
  {
    if (ySize <= 0  ||  values.length == 0) {
      throw new IllegalArgumentException("Need positive sizes!");
    }
    if (values.length % ySize != 0) {
      throw new IllegalArgumentException("Need a multiple of ySize values!");
    }
    this.values = values.clone();
    this.xSize = values.length / ySize;
    this.ySize  = ySize;
  }

  /**
   * Copy constructor.
   * @param array array to copy
   * @throws IllegalArgumentException on negative sizes or if linearized size becoms too large
   */
  public SmallTwoDimensionalCharArray(@NotNull TwoDimensionalCharReadAccess array)
  {
    this(array.sizeX(), array.sizeY(), array::getValueAt);
  }

  /**
   * Copy constructor.
   * @param array array to copy
   * @see #copy()
   */
  private SmallTwoDimensionalCharArray(@NotNull SmallTwoDimensionalCharArray array)
  {
    values = array.values.clone();
    xSize  = array.xSize;
    ySize  = array.ySize;
  }

  /**
   * Invert the row order in X direction.
   */
  public void invertX()
  {
    final char[] tmp = new char[ySize];
    for (int r = 0; r < xSize / 2; ++r) {
      final int pos1 = r * ySize;
      final int pos2 = (xSize - r - 1) * ySize;
      System.arraycopy(values, pos1, tmp, 0, ySize);
      System.arraycopy(values, pos2, values, pos1, ySize);
      System.arraycopy(tmp, 0, values, pos2, ySize);
    }
  }

  /**
   * Invert the row order in Y direction.
   */
  public void invertY()
  {
    for (int r = 0;  r < xSize;  ++r) {
      final int rowStart = r * ySize;
      for (int y = 0;  y < ySize / 2;  ++y) {
        final int pos1 = rowStart + y;
        final int pos2 = rowStart + ySize - y - 1;
        final char value = values[pos1];
        values[pos1] = values[pos2];
        values[pos2] = value;
      }
    }
  }

  /**
   * Check indices and throw {@link IndexOutOfBoundsException} if they are wrong.
   * @param ix X index
   * @param iy Y index
   */
  private void checkIndices(int ix, int iy)
  {
    if (ix >= xSize  ||  ix < 0) {
      throw new IndexOutOfBoundsException(String.format("ix (%d) is out of range [0, %d[!", ix, xSize));
    }
    if (iy >= ySize  ||  iy < 0) {
      throw new IndexOutOfBoundsException(String.format("iy (%d) is out of range [0, %d[!", iy, ySize));
    }
  }

  /**
   * Set the value at the given indexes.
   * @param value  value to set
   * @param ix     X index
   * @param iy     Y index
   */
  public void setValueAt(char value, int ix, int iy)
  {
    checkIndices(ix, iy);
    values[ix * ySize + iy] = value;
  }

  /**
   * Get the value at the given indexes.
   * @param ix  X index
   * @param iy  Y index
   * @return value at given index
   */
  public char getValueAt(int ix, int iy)
  {
    checkIndices(ix, iy);
    return values[ix * ySize + iy];
  }

  @Override
  public int sizeX()
  {
    return xSize;
  }

  @Override
  public int sizeY()
  {
    return ySize;
  }

  /**
   * Get a copy of this array.
   * @return copy of this array
   */
  @NotNull
  public SmallTwoDimensionalCharArray copy()
  {
    return new SmallTwoDimensionalCharArray(this);
  }
}
