// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.IntPredicate2;

/**
 * Two-dimensional array of boolean values optimized for small sizes.
 * <p>
 * This internally maps to a one-dimensional long array which is created faster.
 * Because of the mapping the size of the dimensions is restricted in a way that the
 * number of elements in this array (i.e. the product of its X and Y size) is less
 * than 64 times {@link Integer#MAX_VALUE}.
 * Use {@link TwoDimensionalBooleanArray} if you are not sure to always obey this restriction.
 * <p>
 * Compared to {@link SmallTwoDimensionalBooleanArray} this allows 64 times elements,
 * but has slower access times.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public class SmallTwoDimensionalBitArray
        extends TwoDimensionalBooleanReadAccess.Base
        implements TwoDimensionalBooleanAccess
{
  @NotNull
  private final long[] values;
  private final int xSize;
  private final int ySize;

  /**
   * Constructor.
   * This fills all elements with {@code 0.0} values.
   * @param xSize X size of array
   * @param ySize Y size of array
   * @see #SmallTwoDimensionalBitArray(int, int, IntPredicate2)
   * @throws IllegalArgumentException on negative sizes or if linearized size becoms too large
   */
  public SmallTwoDimensionalBitArray(int xSize, int ySize)
  {
    if (xSize < 0  ||  ySize < 0) {
      throw new IllegalArgumentException("Sizes have to be non-negative!");
    }
    long completeSize = xSize * (long)ySize;
    final long mod = completeSize % 64;
    if (mod != 0) {
      completeSize += 64 - mod;
    }
    if (completeSize / 64 > Integer.MAX_VALUE) {
      throw new IllegalArgumentException("Sizes overflow the range of integer. Use TwoDimensionalBooleanArray instead!");
    }
    values = new long[(int)(completeSize / 64)];
    this.xSize = xSize;
    this.ySize = ySize;
  }

  /**
   * Initializing constructor.
   * @param xSize X size of array
   * @param ySize Y size of array
   * @param filler setter function which provides the value to set for each index combination
   * @see #fillValuesByIndex(IntPredicate2)
   * @throws IllegalArgumentException on negative sizes or if linearized size becoms too large
   */
  public SmallTwoDimensionalBitArray(int xSize, int ySize, @NotNull IntPredicate2 filler)
  {
    this(xSize, ySize);
    fillValuesByIndex(filler);
  }

  /**
   * Constructor.
   * @param ySize  size in y direction
   * @param values values in order, the first {@code ySize} values
   *               will become row 0, the second {@code ySize} values
   *               row 1, and so on. Needs a multiple of ySize values.
   */
  public SmallTwoDimensionalBitArray(int ySize,
                                     boolean... values)
  {
    if (ySize <= 0  ||  values.length == 0) {
      throw new IllegalArgumentException("Need positive sizes!");
    }
    if (values.length % ySize != 0) {
      throw new IllegalArgumentException("Need a multiple of ySize values!");
    }
    long completeSize = values.length;
    final long mod = completeSize % 64;
    if (mod != 0) {
      completeSize += 64 - mod;
    }
    this.values = new long[(int)(completeSize / 64)];
    this.xSize = values.length / ySize;
    this.ySize  = ySize;
    fillByIndex((x, y) -> values[x * ySize + y]);
  }

  /**
   * Copy constructor.
   * @param array array to copy
   * @throws IllegalArgumentException on negative sizes or if linearized size becoms too large
   */
  public SmallTwoDimensionalBitArray(@NotNull TwoDimensionalBooleanReadAccess array)
  {
    this(array.sizeX(), array.sizeY(), array::getValueAt);
  }

  /**
   * Copy constructor.
   * @param array array to copy
   * @see #copy()
   */
  private SmallTwoDimensionalBitArray(@NotNull SmallTwoDimensionalBitArray array)
  {
    values  = array.values.clone();
    xSize   = array.xSize;
    ySize   = array.ySize;
  }

  /**
   * Check indices and throw {@link IndexOutOfBoundsException} if they are wrong.
   * @param ix X index
   * @param iy Y index
   */
  private void checkIndices(int ix, int iy)
  {
    if (ix >= xSize  ||  ix < 0) {
      throw new IndexOutOfBoundsException(String.format("ix (%d) is out of range [0, %d[!", ix, xSize));
    }
    if (iy >= ySize  ||  iy < 0) {
      throw new IndexOutOfBoundsException(String.format("iy (%d) is out of range [0, %d[!", iy, ySize));
    }
  }

  /**
   * Set the value at the given indexes.
   * @param value  value to set
   * @param ix     X index
   * @param iy     Y index
   */
  public void setValueAt(boolean value, int ix, int iy)
  {
    checkIndices(ix, iy);
    final long index = ix * (long)ySize + iy;
    final int longIndex = (int)(index / 64);
    final int longOffset = (int)(index % 64);
    values[longIndex] |= 0x1L << longOffset;
  }

  /**
   * Get the value at the given indexes.
   * @param ix  X index
   * @param iy  Y index
   * @return value at given index
   */
  public boolean getValueAt(int ix, int iy)
  {
    checkIndices(ix, iy);
    final long index = ix * (long)ySize + iy;
    final int longIndex = (int)(index / 64);
    final int longOffset = (int)(index % 64);
    return (values[longIndex] | (0x01L << longOffset)) != 0;
  }

  @Override
  public int sizeX()
  {
    return xSize;
  }

  @Override
  public int sizeY()
  {
    return ySize;
  }

  /**
   * Get a copy of this array.
   * @return copy of this array
   */
  @NotNull
  public SmallTwoDimensionalBitArray copy()
  {
    return new SmallTwoDimensionalBitArray(this);
  }
}
