// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.IntFunction2;

/**
 * Two-dimensional array of objkects values optimized for small sizes.
 * <p>
 * This internally maps to a one-dimensional array which is created faster.
 * Because of the mapping the size of the dimensions is restricted in a way that the
 * number of elements in this array (i.e. the product of its X and Y size) is less
 * than or equal to{@link Integer#MAX_VALUE}. Use {@link TwoDimensionalArray} if you are not
 * sure to always obey this restriction.
 *
 * @param <T> element type of this array
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 26, 2021
 */
public class SmallTwoDimensionalArray<T>
        extends TwoDimensionalReadAccess.Base<T>
        implements TwoDimensionalAccess<T>
{
  @NotNull
  private final T[] values;
  private final int xSize;
  private final int ySize;

  /**
   * Constructor.
   * This fills all elements with {@code 0.0} values.
   * @param xSize X size of array
   * @param ySize Y size of array
   * @see #SmallTwoDimensionalArray(int, int, IntFunction2)
   */
  @SuppressWarnings("unchecked") // the compiler should take care of handling this correctly later on
  public SmallTwoDimensionalArray(int xSize, int ySize)
  {
    if (xSize < 0  ||  ySize < 0) {
      throw new IllegalArgumentException("Sizes have to be non-negative!");
    }
    if (Integer.MAX_VALUE / xSize < ySize) {
      throw new IllegalArgumentException("Sizes overflow the range of integer. Use TwoDimensionalDoubleArray instead!");
    }
    values = (T[])new Object[xSize*ySize];
    this.xSize = xSize;
    this.ySize = ySize;
  }

  /**
   * Initializing constructor.
   * @param xSize X size of array
   * @param ySize Y size of array
   * @param filler setter function which provides the value to set for each index combination
   * @see #fillByIndex(IntFunction2)
   */
  public SmallTwoDimensionalArray(int xSize, int ySize, @NotNull IntFunction2<T> filler)
  {
    this(xSize, ySize);
    fillByIndex(filler);
  }

  /**
   * Constructor.
   * @param ySize  size in y direction
   * @param values values in order, the first {@code ySize} values
   *               will become row 0, the second {@code ySize} values
   *               row 1, and so on. Needs a multiple of ySize values.
   */
  @SafeVarargs // because they are always copied
  @SuppressWarnings("varargs")
  public SmallTwoDimensionalArray(int ySize,
                                  T... values)
  {
    if (ySize <= 0  ||  values.length == 0) {
      throw new IllegalArgumentException("Need positive sizes!");
    }
    if (values.length % ySize != 0) {
      throw new IllegalArgumentException("Need a multiple of ySize values!");
    }
    this.values = values.clone();
    this.xSize = values.length / ySize;
    this.ySize  = ySize;
  }

  /**
   * Copy constructor.
   * @param array array to copy
   */
  public SmallTwoDimensionalArray(@NotNull TwoDimensionalReadAccess<? extends T> array)
  {
    this(array.sizeX(), array.sizeY(), array::getElementAt);
  }

  /**
   * Copy constructor.
   * @param array array to copy
   * @see #copy()
   */
  private SmallTwoDimensionalArray(@NotNull SmallTwoDimensionalArray<? extends T> array)
  {
    values = array.values.clone();
    xSize  = array.xSize;
    ySize  = array.ySize;
  }

  /**
   * Invert the row order in X direction.
   */
  public void invertX()
  {
    final double[] tmp = new double[ySize];
    for (int r = 0; r < xSize / 2; ++r) {
      final int pos1 = r * ySize;
      final int pos2 = (xSize - r - 1) * ySize;
      System.arraycopy(values, pos1, tmp, 0, ySize);
      System.arraycopy(values, pos2, values, pos1, ySize);
      System.arraycopy(tmp, 0, values, pos2, ySize);
    }
  }

  /**
   * Invert the row order in Y direction.
   */
  public void invertY()
  {
    for (int r = 0;  r < xSize;  ++r) {
      final int rowStart = r * ySize;
      for (int y = 0;  y < ySize / 2;  ++y) {
        final int pos1 = rowStart + y;
        final int pos2 = rowStart + ySize - y - 1;
        final T value = values[pos1];
        values[pos1] = values[pos2];
        values[pos2] = value;
      }
    }
  }

  /**
   * Check indices and throw {@link IndexOutOfBoundsException} if they are wrong.
   * @param ix X index
   * @param iy Y index
   */
  private void checkIndices(int ix, int iy)
  {
    if (ix >= xSize  ||  ix < 0) {
      throw new IndexOutOfBoundsException(String.format("ix (%d) is out of range [0, %d[!", ix, xSize));
    }
    if (iy >= ySize  ||  iy < 0) {
      throw new IndexOutOfBoundsException(String.format("iy (%d) is out of range [0, %d[!", iy, ySize));
    }
  }

  @Override
  public void setElementAt(T element, int ix, int iy)
  {
    checkIndices(ix, iy);
    values[ix * ySize + iy] = element;
  }

  @Override
  public T getElementAt(int ix, int iy)
  {
    checkIndices(ix, iy);
    return values[ix * ySize + iy];
  }

  @Override
  public int sizeX()
  {
    return xSize;
  }

  @Override
  public int sizeY()
  {
    return ySize;
  }

  /**
   * Get a copy of this array.
   * @return copy of this array
   */
  @NotNull
  public SmallTwoDimensionalArray<T> copy()
  {
    return new SmallTwoDimensionalArray<T>(this);
  }
}
