// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;

import java.util.Arrays;

/**
 * A multi-index linearizer which keeps some indexes
 * fix and only allows to set the remaining ones.
 * <p>
 * Not public because automatically created when accessing sub arrays
 * and therefore not fit for base arrays.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 01, 2019
 */
class PartlyFixedMultiIndexLinearizer
        implements MultiIndexLinearizer
{
  @NotNull
  private final MultiIndexLinearizer basicLinearizer;
  @NotNull
  private final int[] fixedIndexes;

  private final int[] mapping;
  private final int numElements;

  /**
   * Create a lower-dimensional linearizer by providing some fixed indexes.
   *
   * @param basicLinearizer basic linearizer with n dimensions
   * @param fixedIndexes indexes of sub array,
   *                     negative for open indexes which can still be evalutated
   *                     by the returned linearizer,
   *                     or {@code 0} or positive for fixed indexes. If
   *                     there are not enough indexes the missing ones
   *                     are considered open.
   */
  PartlyFixedMultiIndexLinearizer(@NotNull MultiIndexLinearizer basicLinearizer,
                                  @NotNull int... fixedIndexes)
  {
    if (fixedIndexes.length > basicLinearizer.getNumDimensions()) {
      throw new IllegalArgumentException("Too many fixed values for the given linearizer!");
    }
    this.basicLinearizer = basicLinearizer;
    this.fixedIndexes = new int[basicLinearizer.getNumDimensions()];
    Arrays.fill(this.fixedIndexes, -1);
    System.arraycopy(fixedIndexes, 0, this.fixedIndexes, 0, fixedIndexes.length);

    int unfixed = 0;
    int size = 1;
    final int[] map = new int[this.fixedIndexes.length];
    for (int i = 0;  i < this.fixedIndexes.length;  ++i) {
      final int fix = this.fixedIndexes[i];
      final int dimSize = basicLinearizer.getSize(i);
      if (fix < 0) {
        map[unfixed++] = i;
        size *= dimSize;
      }
      else {
        if (fix >= dimSize) {
          throw new IllegalArgumentException(String.format("Fixed index %d is too large for %d elements!",
                                                           fix, dimSize));
        }
      }
    }

    mapping = Arrays.copyOf(map, unfixed);
    numElements = size;
  }

  @Override
  public int getNumDimensions()
  {
    return mapping.length;
  }

  @Override
  public int getSize(int dim)
  {
    return basicLinearizer.getSize(mapping[dim]);
  }

  @NotNull
  @Override
  public int[] getSizes()
  {
    final int[] result = new int[mapping.length];
    for (int i = 0;  i < result.length;  ++i) {
      result[i] = getSize(i);
    }
    return result;
  }

  @Override
  public int toLinear(@NotNull int... indexes)
  {
    if (indexes.length != getNumDimensions()) {
      throw new IllegalArgumentException(String.format("Incorrect number of indexes for an array with %d dimensions: %d!",
                                                       mapping.length, indexes.length));
    }
    final int[] expanded = new int[fixedIndexes.length];
    int incoming = 0;
    for (int i = 0;  i < expanded.length;  ++i) {
      final int fix = fixedIndexes[i];
      expanded[i] = fix >= 0
              ? fix
              : indexes[incoming++];
    }
    return basicLinearizer.toLinear(expanded);
  }

  @Override
  public long getNumElements()
  {
    return numElements;
  }
}
