// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.Indexable;
import de.caff.generics.function.Function1;
import de.caff.generics.function.Procedure1;
import de.caff.generics.function.Procedure2;

/**
 * A one-dimensional read-only interface.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface OneDimensionalReadAccess<T>
        extends Indexable<T>,
                MultiDimensionalReadAccess<T>
{
  /** Empty 1dimensional read access. Use {@link #empty()} instead. */
  OneDimensionalReadAccess<?> EMPTY = new OneDimensionalReadAccess.Base<Object>()
  {
    @Override
    public Object get(int index)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int size()
    {
      return 0;
    }
  };

  @Override
  default T getElement(int... indexes)
  {
    if (indexes.length != 1) {
      throw new IllegalArgumentException("Invalid number of indexes for 1-dimensional array: "+ indexes.length);
    }
    return get(indexes[0]);
  }

  @Override
  default int getNumDimensions()
  {
    return 1;
  }

  @Override
  default int getSize(int dim)
  {
    if (dim != 0) {
      throw new IllegalArgumentException("Invalid dimension for 1-dimensional array: "+ dim);
    }
    return size();
  }

  @NotNull
  @Override
  default int[] getSizes()
  {
    return new int[] { size() };
  }

  @Override
  default long getNumElements()
  {
    return size();
  }

  /**
   * Get an independent copy.
   * @return indpendent copy
   */
  @NotNull
  default OneDimensionalReadAccess<T> copy()
  {
    return new OneDimensionalArray<>(this);
  }

  /**
   * Get a view of this 1-dimensional access with converted elements.
   * @param conv    converted applied to the elements
   * @param <TOUT>  element type of the returned view
   * @return view of this 1-dimensional read access with converted elements
   */
  @NotNull
  default <TOUT> OneDimensionalReadAccess<TOUT> view(@NotNull Function1<TOUT, ? super T> conv)
  {
    return new OneDimensionalReadAccess<TOUT>()
    {
      @Override
      public int size()
      {
        return OneDimensionalReadAccess.this.size();
      }

      @Override
      public TOUT get(int index)
      {
        return conv.apply(OneDimensionalReadAccess.this.get(index));
      }
    };
  }

  @Override
  default void visitAll(@NotNull Procedure2<? super T, int[]> visitor)
  {
    final int sz = size();
    for (int i = 0;  i < sz;  ++i) {
      visitor.apply(get(i), new int[] { i });
    }
  }

  @Override
  default void visitAll(@NotNull Procedure1<? super T> visitor)
  {
    final int sz = size();
    for (int i = 0;  i < sz;  ++i) {
      visitor.apply(get(i));
    }
  }

  /**
   * Make this read access usable as a base class with default Object methods implemented.
   * @return base version of this access
   */
  @NotNull
  @Override
  default Base<T> asBase()
  {
    return new Base<T>() {
      @Override
      public int size()
      {
        return OneDimensionalReadAccess.this.size();
      }

      @Override
      public T get(int index)
      {
        return OneDimensionalReadAccess.this.get(index);
      }
    };
  }

  /**
   * Get an empty 1dimensional read access.
   * @param <E> element type
   * @return {@link #EMPTY} in a form the compiler accepts
   */
  @NotNull
  @SuppressWarnings("unchecked") // as EMPTY is immutable
  static <E> OneDimensionalReadAccess<E> empty()
  {
    return (OneDimensionalReadAccess<E>)EMPTY;
  }

  /**
   * Get a 1dimensional read access for a single element
   * @param singleElement single element
   * @return 1dimensional read access with size 1,
   *         containing just the given single element
   * @param <E> element type
   */
  @NotNull
  static <E> OneDimensionalReadAccess<E> singleton(E singleElement)
  {
    return new OneDimensionalReadAccess<E>()
    {
      @Override
      public E get(int index)
      {
        if (index != 0) {
          throw new IndexOutOfBoundsException();
        }
        return singleElement;
      }

      @Override
      public int size()
      {
        return 1;
      }
    };
  }

  /**
   * Abstract base class which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}.
   * @param <TT> element type
   * @see #asBase()
   */
  abstract class Base<TT> extends Indexable.Base<TT>
          implements OneDimensionalReadAccess<TT>
  {
    @NotNull
    public OneDimensionalReadAccess.Base<TT> asBase()
    {
      return this;
    }
  }
}
