// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.LongOperator1;

import java.util.function.IntToLongFunction;

/**
 * Read/write access for 1-dimensional long arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface OneDimensionalLongAccess
        extends OneDimensionalLongReadAccess,
                OneDimensionalAccess<Long>
{
  /** Empty 1dimensional long access. */
  OneDimensionalLongAccess EMPTY = new OneDimensionalLongAccess()
  {
    @Override
    public void setValueAt(long value, int index)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public long getValueAt(int index)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int size()
    {
      return 0;
    }
  };

  /**
   * Set the value at the given index.
   * <p>
   * This is usually more efficient than
   * {@link #set(int, Long)} or {@link #setElement(Object, int...)}.
   *
   * @param value value to set
   * @param index index of value
   */
  void setValueAt(long value, int index);

  /**
   * Change the value at the given indexes.
   * @param operator operator to apply to the value
   * @param index index of value
   */
  default void changeValueAt(@NotNull LongOperator1 operator, int index)
  {
    setValueAt(operator.applyAsLong(getValueAt(index)), index);
  }

  /**
   * Change all values by applying an operator.
   * @param operator operator to apply
   */
  default void changeAllValues(@NotNull LongOperator1 operator)
  {
    final int sz = size();
    for (int x = 0;  x < sz;  ++x) {
      changeValueAt(operator, x);
    }
  }

  /**
   * Set all values by their index.
   * This will call the setter for all indexes and set the value at the given index
   * from the return value of the setter.
   * @param setter setter which gets the index and returns the value to set for that index
   */
  default void fillValuesByIndex(@NotNull IntToLongFunction setter)
  {
    final int sz = size();
    for (int x = 0;  x < sz;  ++x) {
      setValueAt(setter.applyAsLong(x), x);
    }
  }

  @Override
  default void set(int index, @NotNull Long elem)
  {
    setValueAt(elem, index);
  }
}
