// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.Indexable;
import de.caff.generics.IntIndexable;

import java.util.Collection;
import java.util.function.IntUnaryOperator;

/**
 * A 1-dimensional array of int values.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public class OneDimensionalIntArray
        extends OneDimensionalIntReadAccess.Base
        implements OneDimensionalIntAccess
{
  @NotNull
  private final int[] values;

  /**
   * Constructor.
   * @param values values of this array
   */
  public OneDimensionalIntArray(@NotNull int... values)
  {
    this.values = values.clone();
  }

  /**
   * Constructor.
   * This initializes all elements with {@code 0},
   * @param numElements number of elements in this array
   * @see #OneDimensionalIntArray(int, IntUnaryOperator)
   */
  public OneDimensionalIntArray(int numElements)
  {
    this.values = new int[numElements];
  }

  /**
   * Initializing constructor.
   * @param numElements number of elements in this array
   * @param filler      filler which is called once for each element with its index
   *                    and expected to return the value to be set for that index
   */
  public OneDimensionalIntArray(int numElements, @NotNull IntUnaryOperator filler)
  {
    this(numElements);
    fillValuesByIndex(filler);
  }

  /**
   * Copy constructor.
   * @param source source to copy
   * @throws NullPointerException if source contains {@code null} values
   */
  public OneDimensionalIntArray(@NotNull Indexable<Integer> source)
  {
    this(source.size(), source::get);
  }

  /**
   * Copy constructor.
   * @param source source to copy
   */
  public OneDimensionalIntArray(@NotNull IntIndexable source)
  {
    this(source.size(), source::get);
  }

  /**
   * Constructor.
   * @param elements elements to copy
   * @throws NullPointerException if elements contain {@code null} values
   */
  public OneDimensionalIntArray(@NotNull Collection<? extends Integer> elements)
  {
    this(elements.size());
    int idx = 0;
    for (Integer elem : elements) {
      values[idx++] = elem;
    }
  }

  /**
   * Copy constructor.
   * @param source source to copy
   */
  public OneDimensionalIntArray(@NotNull OneDimensionalIntArray source)
  {
    this(source.values);
  }

  /**
   * Copy constructor.
   * @param array sourve array to copy
   */
  public OneDimensionalIntArray(@NotNull MultiDimensionalIntArray array)
  {
    if (array.getNumDimensions() != 1) {
      throw new IllegalArgumentException("Cannot only copy from 1-dimensional arrays!");
    }
    values = new int[array.getSize(0)];
    for (int v = values.length - 1;  v >= 0;  --v) {
      values[v] = array.getValue(v);
    }
  }

  /**
   * Set the value at the given index.
   *
   * @param value value to set
   * @param index index of value
   */
  public void setValueAt(int value, int index)
  {
    values[index] = value;
  }

  /**
   * Get the value at the given index.
   *
   * @param index index of value
   * @return value at given index
   */
  public int getValueAt(int index)
  {
    return values[index];
  }

  @Override
  public int size()
  {
    return values.length;
  }
}
