// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.DoubleIndexable;
import de.caff.generics.Indexable;

import java.util.Collection;
import java.util.function.IntToDoubleFunction;

/**
 * A 1-dimensional array of double values.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public class OneDimensionalDoubleArray
        extends OneDimensionalDoubleReadAccess.Base
        implements OneDimensionalDoubleAccess
{
  @NotNull
  private final double[] values;

  /**
   * Constructor.
   * @param values values of this array
   */
  public OneDimensionalDoubleArray(@NotNull double... values)
  {
    this.values = values.clone();
  }

  /**
   * Constructor.
   * This initializes all elements with {@code 0.0}.
   * @param numElements number of elements in this array
   */
  public OneDimensionalDoubleArray(int numElements)
  {
    this.values = new double[numElements];
  }

  /**
   * Initializing constructor.
   * @param numElements number of elements in this array
   * @param filler      filler which is called once for each element with its index
   *                    and expected to return the value to be set for that index
   */
  public OneDimensionalDoubleArray(int numElements, @NotNull IntToDoubleFunction filler)
  {
    this(numElements);
    fillValuesByIndex(filler);
  }

  /**
   * Copy constructor.
   * @param source source to copy
   * @throws NullPointerException if source contains {@code null} values
   */
  public OneDimensionalDoubleArray(@NotNull Indexable<Double> source)
  {
    this(source.size(), source::get);
  }

  /**
   * Copy constructor.
   * @param source source to copy
   * @throws NullPointerException if source contains {@code null} values
   */
  public OneDimensionalDoubleArray(@NotNull DoubleIndexable source)
  {
    this(source.size(), source::get);
  }

  /**
   * Constructor.
   * @param elements elements to copy
   * @throws NullPointerException if elements contain {@code null} values
   */
  public OneDimensionalDoubleArray(@NotNull Collection<Double> elements)
  {
    this(elements.size());
    int idx = 0;
    for (Double elem : elements) {
      values[idx++] = elem;
    }
  }

  /**
   * Copy constructor.
   * @param source source to copy
   */
  public OneDimensionalDoubleArray(@NotNull OneDimensionalDoubleArray source)
  {
    this(source.values);
  }

  /**
   * Copy constructor.
   * @param array sourve array to copy
   */
  public OneDimensionalDoubleArray(@NotNull MultiDimensionalDoubleArray array)
  {
    if (array.getNumDimensions() != 1) {
      throw new IllegalArgumentException("Cannot only copy from 1-dimensional arrays!");
    }
    values = new double[array.getSize(0)];
    for (int v = values.length - 1;  v >= 0;  --v) {
      values[v] = array.getValue(v);
    }
  }

  /**
   * Set the value at the given index.
   *
   * @param value value to set
   * @param index index of value
   */
  public void setValueAt(double value, int index)
  {
    values[index] = value;
  }

  /**
   * Get the value at the given index.
   *
   * @param index index of value
   * @return value at given index
   */
  public double getValueAt(int index)
  {
    return values[index];
  }

  @Override
  public int size()
  {
    return values.length;
  }
}
