// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;

/**
 * Read access for 1-dimensional byte arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface OneDimensionalByteReadAccess
        extends OneDimensionalReadAccess<Byte>
{
  /** Empty 1dimensional byte read access. */
  OneDimensionalByteReadAccess EMPTY = new OneDimensionalByteReadAccess.Base()
  {
    @Override
    public byte getValueAt(int index)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int size()
    {
      return 0;
    }
  };

  /**
   * Get the value at the given index.
   * <p>
   * This is usually more efficient then
   * {@link #get(int)} and {@link #getElement(int...)}.
   *
   * @param index index of requested value
   * @return value at the given index
   */
  byte getValueAt(int index);

  @Override
  @NotNull
  default Byte get(int index)
  {
    return getValueAt(index);
  }

  /**
   * Make this read access usable as a base class with default Object methods implemented.
   * @return base version of this access
   */
  @NotNull
  @Override
  default OneDimensionalByteReadAccess.Base asBase()
  {
    return new OneDimensionalByteReadAccess.Base()
    {
      @Override
      public byte getValueAt(int index)
      {
        return OneDimensionalByteReadAccess.this.getValueAt(index);
      }

      @Override
      public int size()
      {
        return OneDimensionalByteReadAccess.this.size();
      }
    };
  }

  /**
   * Create an 1dimensional byte read access with only 1 element.
   * @param singleElement the single element
   * @return 1dimensional read access with the given single element
   */
  @NotNull
  static OneDimensionalByteReadAccess singleton(byte singleElement)
  {
    return new OneDimensionalByteReadAccess.Base()
    {
      @Override
      public byte getValueAt(int index)
      {
        if (index != 0) {
          throw new IndexOutOfBoundsException();
        }
        return singleElement;
      }

      @Override
      public int size()
      {
        return 1;
      }
    };
  }

  /**
   * Abstract base class implementing this interface which provides useful implementations
   * for {@link Object#equals(Object)}, {@link Object#hashCode()},
   * {@link Object#toString()}.
   * @see #asBase()
   */
  abstract class Base extends OneDimensionalReadAccess.Base<Byte>
          implements OneDimensionalByteReadAccess
  {
    @NotNull
    @Override
    public OneDimensionalByteReadAccess.Base asBase()
    {
      return this;
    }
  }
}
