// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.Indexable;
import de.caff.generics.Types;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.IntFunction;

/**
 * Standard implementation for a 1-dimensional access.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public class OneDimensionalArray<T>
        extends OneDimensionalReadAccess.Base<T>
        implements OneDimensionalAccess<T>
{
  @NotNull
  private final List<T> elements;

  /**
   * Constructor.
   * @param elements elements of this array
   */
  public OneDimensionalArray(@NotNull Collection<T> elements)
  {
    this.elements = new ArrayList<>(elements);
  }

  /**
   * Constructor.
   * @param elements elements of this array
   */
  public OneDimensionalArray(@NotNull Indexable<T> elements)
  {
    this.elements = Types.map(new ArrayList<>(elements.size()),
                              elements, e -> e);
  }

  /**
   * Constructor.
   * This will fill all elements with {@code null}.
   * @param size size of this array
   */
  public OneDimensionalArray(int size)
  {
    this(size, v -> null);
  }

  /**
   * Initializing Constructor.
   * @param size   size of this array
   * @param filler filler which is called with for each element with its index
   *               and expected to return the value at that index
   */
  public OneDimensionalArray(int size, @NotNull IntFunction<? extends T> filler)
  {
    elements = new ArrayList<>(size);
    for (int i = 0;  i < size;  ++i) {
      elements.add(filler.apply(i));
    }
  }

  /**
   * Constructor.
   * @param elements elements of this array
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  public OneDimensionalArray(@NotNull T... elements)
  {
    this(Types.asList(elements));
  }

  /**
   * Constructor.
   * @param elements elements of this array.
   */
  public OneDimensionalArray(@NotNull Iterable<T> elements)
  {
    final ArrayList<T> list = Types.map(new ArrayList<>(),
                                        elements, e -> e);
    list.trimToSize();
    this.elements = list;
  }

  @NotNull
  @Override
  public OneDimensionalAccess<T> copy()
  {
    return new OneDimensionalArray<>(this);
  }

  @Override
  public void set(int index, T elem)
  {
    elements.set(index, elem);
  }

  @Override
  public int size()
  {
    return elements.size();
  }

  @Override
  public T get(int index)
  {
    return elements.get(index);
  }
}
