// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.MutableIndexable;

import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.IntFunction;

/**
 * A fixed-size one-dimensional array of items.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public interface OneDimensionalAccess<T>
        extends MutableIndexable<T>,
                MultiDimensionalAccess<T>,
                OneDimensionalReadAccess<T>
{
  /** Empty 1dimensional access. Use {@link #empty()} instead. */
  OneDimensionalAccess<?> EMPTY = new OneDimensionalAccess<Object>()
  {
    @Override
    public void set(int index, Object elem)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public Object get(int index)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int size()
    {
      return 0;
    }
  };

  /**
   * Get an independent copy.
   * @return independent copy
   */
  @NotNull
  default OneDimensionalAccess<T> copy()
  {
    return new OneDimensionalArray<>(this);
  }

  /**
   * Set the value at the given index.
   * This provides the same functionality as {@link #set(int, Object)},
   * but has the same parameter order as the raw sister implementations of
   * this interface.
   * @param value value to set
   * @param index index where to set the value
   */
  default void setElementAt(T value, int index)
  {
    set(index, value);
  }

  @Override
  default void setElement(T value, int... indexes)
  {
    if (indexes.length != 1) {
      throw new IllegalArgumentException("Invalid number of indexes for 1-dimensional array: "+ indexes.length);
    }
    set(indexes[0], value);
  }

  @Override
  default void changeAll(@NotNull BiFunction<? super T, int[], ? extends T> operator)
  {
    final int sz = size();
    for (int i = 0;  i < sz;  ++i) {
      set(i, operator.apply(get(i), new int[] { i }));
    }
  }

  @Override
  default void changeAll(@NotNull Function<? super T, ? extends T> operator)
  {
    final int sz = size();
    for (int i = 0;  i < sz;  ++i) {
      set(i, operator.apply(get(i)));
    }
  }

  /**
   * Set all values by their index.
   * This will call the setter for all indexes and set the value at the given index
   * from the return value of the setter.
   * @param setter setter which gets the index and returns the value to set for that index
   */
  default void fillByIndex(@NotNull IntFunction<? extends T> setter)
  {
    final int sz = size();
    for (int x = 0;  x < sz;  ++x) {
      setElementAt(setter.apply(x), x);
    }
  }

  /**
   * Get an empty 1dimensional access.
   * @param <E> element type
   * @return {@link #EMPTY} in a form the compiler accepts
   */
  @NotNull
  @SuppressWarnings("unchecked") // as EMPTY is immutable
  static <E> OneDimensionalAccess<E> empty()
  {
    return (OneDimensionalAccess<E>)EMPTY;
  }
}
