// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.Indexable;

/**
 * Accessor for a linear indexed item
 * which provides a multi-dimensional interface.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 01, 2019
 */
public interface MultiIndexLinearizer
        extends MultiDimensional
{
  /**
   * Special index used for open indexes.
   * This has to be used for {@link #sub(int...)}
   * parameters to define an open index.
   */
  int OPEN = Integer.MIN_VALUE;

  /**
   * Get the linear index which represents the given multi-dimensional index.
   * Please note that the returned values may fall outside of the range
   * defined by {@link #getNumElements()} when this linearizer only
   * represents a part of a larger-dimensional multi-index.
   * <p>
   * Good practice is to check the incoming index whether they fulfill
   * their bounds. Use {@link #checkIndexes(int[], int...)} for this.
   *
   * @param indexes {@link MultiDimensional#getNumDimensions()} indexes inside the ranges defined
   *                by {@link #getSizes()}
   * @return linearized index
   */
  int toLinear(int... indexes);

  /**
   * Get an indexer which indexes only a part of this
   * linearizer which is defined by keeping some indexes fix.
   * @param fixedIndexes non-negative values define fixed indexes of sub array,
   *                     {@link #OPEN} or other negative values are used
   *                     for indexes which can still be evalutated
   *                     by the returned linearizer
   * @return tweaked linearizer which accesses only the defined regions
   */
  default MultiIndexLinearizer sub(int... fixedIndexes)
  {
    for (int index : fixedIndexes) {
      if (index >= 0) {
        return new PartlyFixedMultiIndexLinearizer(this, fixedIndexes);
      }
    }
    return this;
  }

  /**
   * Get a sequencer will create a multi-index
   * by changing the highest dimension most fast.
   * <p>
   * For one-dimensional indexes it behaves the same as {@link #getLowFastSequencer()}.
   * <p>
   * Highest fast is the standard way in which multi-indexes are created,
   * so using this sequencer on a non-sub indexer will iterate over the underlying
   * sequence in its natural order allowing for caching effects.
   *
   * @return sequencer which changes the highest dimension fast, and the lowest dimension slow.
   *         It works on the same set of dimensions as this linear indexer.
   */
  @NotNull
  default Sequencer getHighFastSequencer()
  {
    return new BasicSequencer(this)
    {
      @NotNull
      @Override
      protected int[] get(int index, @NotNull int[] sizes)
      {
        final int[] result = new int[sizes.length];
        for (int i = result.length - 1;  i >= 0;  --i) {
          final int s = sizes[i];
          result[i] = index % s;
          index /= s;
        }
        return result;
      }
    };
  }

  /**
   * Get a sequencer will create a multi-index
   * by changing the lowest dimension most fast.
   * For one-dimensional indexes it behaves the same as {@link #getHighFastSequencer()}.
   * @return sequencer which changes the highest dimension fast, and the lowest dimension slow.
   *         It works on the same set of dimensions as this linear indexer.
   */
  @NotNull
  default Sequencer getLowFastSequencer()
  {
    return new BasicSequencer(this)
    {
      @NotNull
      @Override
      protected int[] get(int index, @NotNull int[] sizes)
      {
        final int[] result = new int[sizes.length];
        for (int i = 0;  i < result.length;  ++i) {
          final int s = sizes[i];
          result[i] = index % s;
          index /= s;
        }
        return result;
      }
    };
  }

  /**
   * Check indexes whether they fulfill the basic restrictions.
   * This checks
   * <ul>
   *   <li>whether both arrays have the same size</li>
   *   <li>
   *     whether each index falls into the bounds defined
   *     by the sizes, i.e. whether it is positive and
   *     less than the size of the given dimension.
   *   </li>
   * </ul>
   * @param sizes   sizes of the dimensions of an array
   * @param indexes indexes into the dimensions of an array
   * @throws IllegalArgumentException if the lengths of both array differ
   * @throws IndexOutOfBoundsException if the indexes don't fall into the bounds
   */
  static void checkIndexes(int[] sizes, int... indexes)
  {
    if (indexes.length != sizes.length) {
      throw new IllegalArgumentException(String.format("Incorrect number of indexes for an array with %d dimensions: %d!",
                                                       sizes.length, indexes.length));
    }
    for (int i = sizes.length - 1;  i >= 0;  --i) {
      checkIndex(i, sizes[i], indexes[i]);
    }
  }

  /**
   * Check a single index.
   * @param dim   dimension of the index
   * @param size  size of the dimension
   * @param index index
   * @throws IndexOutOfBoundsException if the indexes don't fall into the bounds
   */
  static void checkIndex(int dim, int size, int index)
  {
    if (index < 0 || index >= size) {
      throw new IndexOutOfBoundsException(String.format("Index #%d is out of bounds (size is %d): %d!",
                                                        dim, index, size));
    }

  }

  /**
   * A sequencer creates a multi-index from a linear index.
   * <p>
   * Basically it does the inverse of {@link MultiIndexLinearizer#toLinear(int...)}
   * method, but only for basic {@link MultiIndexLinearizer}s and not for
   * {@link MultiIndexLinearizer#sub(int...) sub index linearizers}.
   * The returned index can be used to iterate over a multi index
   * in a defined way. It is especially useful for one-dimensional
   * arrays.
   */
  interface Sequencer
          extends Indexable<int[]>,
                  MultiDimensional
  {
  }

  /**
   * Basic implementation of a Sequencer.
   * <p>
   * This provides everything but the actual sequencing.
   */
  abstract class BasicSequencer
          implements Sequencer
  {
    @NotNull
    private final MultiIndexLinearizer indexLinearizer;
    @NotNull
    private final int[] sizes;

    /**
     * Constructor.
     * @param indexLinearizer multi-index linearizer on which this sequencer operates
     */
    public BasicSequencer(@NotNull MultiIndexLinearizer indexLinearizer)
    {
      this.indexLinearizer = indexLinearizer;
      sizes = indexLinearizer.getSizes();
    }

    @Override
    public int getNumDimensions()
    {
      return indexLinearizer.getNumDimensions();
    }

    @Override
    public int getSize(int dim)
    {
      return indexLinearizer.getSize(dim);
    }

    @Override
    public long getNumElements()
    {
      return indexLinearizer.getNumElements();
    }

    @NotNull
    @Override
    public int[] getSizes()
    {
      return sizes.clone();
    }

    @Override
    public int size()
    {
      return (int)getNumElements();
    }

    @Override
    @NotNull
    public int[] get(int index)
    {
      final int numElements = size();
      if (index < 0  ||  index >= numElements) {
        throw new IndexOutOfBoundsException(String.format("Index is out of bounds (0 to %d): %d!",
                                                          numElements, index));
      }
      return get(index, sizes);
    }

    /**
     * Get the multi-indexes for the given sequence index.
     * @param index linear sequence index between  {@code 0} (included)
     *              and {@link #getNumElements()} (excluded)
     * @param sizes sizes of the dimensions (do not change!)
     * @return multi-indexes for access the {@code index}th element
     */
    @NotNull
    protected abstract int[] get(int index,
                                 @NotNull int[] sizes);
  }

}
