// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.Empty;
import de.caff.generics.Types;
import de.caff.generics.function.Procedure1;
import de.caff.generics.function.Procedure2;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * Multi-dimensional read access.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 06, 2019
 */
public interface MultiDimensionalReadAccess<T>
        extends MultiDimensional
{
  /** Empty multi-dimensional read access for 0 dimensions. Use {@code MultiDimensionalReadAccess.empty(0)} instead. */
  MultiDimensionalReadAccess<?> EMPTY0 = new MultiDimensionalReadAccess<Object>()
  {
    @Override
    public Object getElement(int... indexes)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int getNumDimensions()
    {
      return 0;
    }

    @Override
    public int getSize(int dim)
    {
      throw new IllegalArgumentException("No dimension "+dim);
    }

    @NotNull
    @Override
    public int[] getSizes()
    {
      return Empty.INT_ARRAY;
    }

    @Override
    public long getNumElements()
    {
      return 0;
    }

    @Override
    public void visitAll(@NotNull Procedure2<? super Object, int[]> visitor)
    {
    }

    @Override
    public void visitAll(@NotNull Procedure1<? super Object> visitor)
    {
    }

    @NotNull
    @Override
    public Iterable<Object> linearized()
    {
      return Types.emptyIterable();
    }
  };

  /** Empty multi-dimensional read access for 1 dimension. Use {@code MultiDimensionalReadAccess.empty(1)} instead. */
  MultiDimensionalReadAccess<?> EMPTY1 = new MultiDimensionalReadAccess<Object>()
  {
    @Override
    public Object getElement(int... indexes)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int getNumDimensions()
    {
      return 1;
    }

    @Override
    public int getSize(int dim)
    {
      switch (dim) {
      case 0:
        return 0;
      default:
        throw new IllegalArgumentException("No dimension "+dim);
      }
    }

    @NotNull
    @Override
    public int[] getSizes()
    {
      return new int[1];
    }

    @Override
    public long getNumElements()
    {
      return 0;
    }

    @Override
    public void visitAll(@NotNull Procedure2<? super Object, int[]> visitor)
    {
    }

    @Override
    public void visitAll(@NotNull Procedure1<? super Object> visitor)
    {
    }

    @NotNull
    @Override
    public Iterable<Object> linearized()
    {
      return Types.emptyIterable();
    }
  };

  /** Empty multi-dimensional read access for 2 dimensions. Use {@code MultiDimensionalReadAccess.empty(2)} instead. */
  MultiDimensionalReadAccess<?> EMPTY2 = new MultiDimensionalReadAccess<Object>()
  {
    @Override
    public Object getElement(int... indexes)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int getNumDimensions()
    {
      return 2;
    }

    @Override
    public int getSize(int dim)
    {
      switch (dim) {
      case 0:
      case 1:
        return 0;
      default:
        throw new IllegalArgumentException("No dimension "+dim);
      }
    }

    @NotNull
    @Override
    public int[] getSizes()
    {
      return new int[2];
    }

    @Override
    public long getNumElements()
    {
      return 0;
    }

    @Override
    public void visitAll(@NotNull Procedure2<? super Object, int[]> visitor)
    {
    }

    @Override
    public void visitAll(@NotNull Procedure1<? super Object> visitor)
    {
    }

    @NotNull
    @Override
    public Iterable<Object> linearized()
    {
      return Types.emptyIterable();
    }
  };

  /** Empty multi-dimensional read access for 3 dimensions. Use {@code MultiDimensionalReadAccess.empty(3)} instead. */
  MultiDimensionalReadAccess<?> EMPTY3 = new MultiDimensionalReadAccess<Object>()
  {
    @Override
    public Object getElement(int... indexes)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int getNumDimensions()
    {
      return 3;
    }

    @Override
    public int getSize(int dim)
    {
      switch (dim) {
      case 0:
      case 1:
      case 2:
        return 0;
      default:
        throw new IllegalArgumentException("No dimension "+dim);
      }
    }

    @NotNull
    @Override
    public int[] getSizes()
    {
      return new int[3];
    }

    @Override
    public long getNumElements()
    {
      return 0;
    }

    @Override
    public void visitAll(@NotNull Procedure2<? super Object, int[]> visitor)
    {
    }

    @Override
    public void visitAll(@NotNull Procedure1<? super Object> visitor)
    {
    }

    @NotNull
    @Override
    public Iterable<Object> linearized()
    {
      return Types.emptyIterable();
    }
  };

  /** Empty multi-dimensional read access for 4 dimensions. Use {@code MultiDimensionalReadAccess.empty(4)} instead. */
  MultiDimensionalReadAccess<?> EMPTY4 = new MultiDimensionalReadAccess<Object>()
  {
    @Override
    public Object getElement(int... indexes)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int getNumDimensions()
    {
      return 4;
    }

    @Override
    public int getSize(int dim)
    {
      switch (dim) {
      case 0:
      case 1:
      case 2:
      case 3:
        return 0;
      default:
        throw new IllegalArgumentException("No dimension "+dim);
      }
    }

    @NotNull
    @Override
    public int[] getSizes()
    {
      return new int[4];
    }

    @Override
    public long getNumElements()
    {
      return 0;
    }

    @Override
    public void visitAll(@NotNull Procedure2<? super Object, int[]> visitor)
    {
    }

    @Override
    public void visitAll(@NotNull Procedure1<? super Object> visitor)
    {
    }

    @NotNull
    @Override
    public Iterable<Object> linearized()
    {
      return Types.emptyIterable();
    }
  };

  /** Empty multi-dimensional read access for 4 dimensions. Use {@code MultiDimensionalReadAccess.empty(5)} instead. */
  MultiDimensionalReadAccess<?> EMPTY5 = new MultiDimensionalReadAccess<Object>()
  {
    @Override
    public Object getElement(int... indexes)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int getNumDimensions()
    {
      return 5;
    }

    @Override
    public int getSize(int dim)
    {
      switch (dim) {
      case 0:
      case 1:
      case 2:
      case 3:
      case 4:
        return 0;
      default:
        throw new IllegalArgumentException("No dimension "+dim);
      }
    }

    @NotNull
    @Override
    public int[] getSizes()
    {
      return new int[5];
    }

    @Override
    public long getNumElements()
    {
      return 0;
    }

    @Override
    public void visitAll(@NotNull Procedure2<? super Object, int[]> visitor)
    {
    }

    @Override
    public void visitAll(@NotNull Procedure1<? super Object> visitor)
    {
    }

    @NotNull
    @Override
    public Iterable<Object> linearized()
    {
      return Types.emptyIterable();
    }
  };

  /** Empty multi-dimensional read access for 4 dimensions. Use {@code MultiDimensionalReadAccess.empty(6)} instead. */
  MultiDimensionalReadAccess<?> EMPTY6 = new MultiDimensionalReadAccess<Object>()
  {
    @Override
    public Object getElement(int... indexes)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int getNumDimensions()
    {
      return 6;
    }

    @Override
    public int getSize(int dim)
    {
      switch (dim) {
      case 0:
      case 1:
      case 2:
      case 3:
      case 4:
      case 5:
        return 0;
      default:
        throw new IllegalArgumentException("No dimension "+dim);
      }
    }

    @NotNull
    @Override
    public int[] getSizes()
    {
      return new int[6];
    }

    @Override
    public long getNumElements()
    {
      return 0;
    }

    @Override
    public void visitAll(@NotNull Procedure2<? super Object, int[]> visitor)
    {
    }

    @Override
    public void visitAll(@NotNull Procedure1<? super Object> visitor)
    {
    }

    @NotNull
    @Override
    public Iterable<Object> linearized()
    {
      return Types.emptyIterable();
    }
  };

  /** Empty multi-dimensional read access for 4 dimensions. Use {@code MultiDimensionalReadAccess.empty(7)} instead. */
  MultiDimensionalReadAccess<?> EMPTY7 = new MultiDimensionalReadAccess<Object>()
  {
    @Override
    public Object getElement(int... indexes)
    {
      throw new IndexOutOfBoundsException();
    }

    @Override
    public int getNumDimensions()
    {
      return 7;
    }

    @Override
    public int getSize(int dim)
    {
      switch (dim) {
      case 0:
      case 1:
      case 2:
      case 3:
      case 4:
      case 5:
      case 6:
        return 0;
      default:
        throw new IllegalArgumentException("No dimension "+dim);
      }
    }

    @NotNull
    @Override
    public int[] getSizes()
    {
      return new int[7];
    }

    @Override
    public long getNumElements()
    {
      return 0;
    }

    @Override
    public void visitAll(@NotNull Procedure2<? super Object, int[]> visitor)
    {
    }

    @Override
    public void visitAll(@NotNull Procedure1<? super Object> visitor)
    {
    }

    @NotNull
    @Override
    public Iterable<Object> linearized()
    {
      return Types.emptyIterable();
    }
  };

  /**
   * Get the element located at the given indexes.
   * @param indexes {@link MultiDimensional#getNumDimensions()} indexes inside the bounds
   *                defined by {@link MultiDimensional#getSizes()}
   * @return element located at the given indexes
   */
  T getElement(int... indexes);

  /**
   * Visit all elements of this multi-dimensional array.
   * @param visitor visitor called with an array element and its multi-index
   */
  default void visitAll(@NotNull Procedure2<? super T, int[]> visitor)
  {
    final int[] sizes = getSizes();
    final int length = sizes.length;
    if (length == 0) {
      return;
    }
    final int[] run = new int[length];
    final int last = length - 1;
    int carry = last;
    while (carry >= 0) {
      visitor.apply(getElement(run), run.clone());
      for (carry = last;  carry >= 0;  --carry) {
        if (++run[carry] < sizes[carry]) {
          break;
        }
        run[carry] = 0;
      }
    }
  }

  /**
   * Visit all elements in this multi-dimensional array.
   * @param visitor element visitor called once with each element
   */
  default void visitAll(@NotNull Procedure1<? super T> visitor)
  {
    final int[] sizes = getSizes();
    final int length = sizes.length;
    if (length == 0) {
      return;
    }
    final int[] run = new int[length];
    final int last = length - 1;
    int carry = last;
    while (carry >= 0) {
      visitor.apply(getElement(run));
      for (carry = last;  carry >= 0;  --carry) {
        if (++run[carry] < sizes[carry]) {
          break;
        }
        run[carry] = 0;
      }
    }
  }

  /**
   * Get an iterable over all elements of this multi-dimensional access.
   * @return iterable which iterates over all elements
   */
  @NotNull
  default Iterable<T> linearized()
  {
    final int[] sizes = getSizes();
    final int length = sizes.length;
    if (length == 0) {
      return Types.emptyIterable();
    }
    final int last = length - 1;
    return () -> new Iterator<T>()
    {
      private final int[] run = new int[length];
      private int carry = last;

      @Override
      public boolean hasNext()
      {
        return carry >= 0;
      }

      @Override
      public T next()
      {
        if (carry < 0) {
          throw new NoSuchElementException();
        }
        final T elem = getElement(run);
        for (carry = last;  carry >= 0;  --carry) {
          if (++run[carry] < sizes[carry]) {
            break;
          }
          run[carry] = 0;
        }
        return elem;
      }
    };
  }

  @NotNull
  @SuppressWarnings("unchecked") // as all EMPTY* are immutable
  static <E> MultiDimensionalReadAccess<E> empty(int numDimensions)
  {
    if (numDimensions < 0) {
      throw new IllegalArgumentException("numDimensions has to be non-negative, but is "+numDimensions);
    }
    switch (numDimensions) {
    case 0:
      return (MultiDimensionalReadAccess<E>)EMPTY0;
    case 1:
      return (MultiDimensionalReadAccess<E>)EMPTY1;
    case 2:
      return (MultiDimensionalReadAccess<E>)EMPTY2;
    case 3:
      return (MultiDimensionalReadAccess<E>)EMPTY3;
    case 4:
      return (MultiDimensionalReadAccess<E>)EMPTY4;
    case 5:
      return (MultiDimensionalReadAccess<E>)EMPTY5;
    case 6:
      return (MultiDimensionalReadAccess<E>)EMPTY6;
    case 7:
      return (MultiDimensionalReadAccess<E>)EMPTY7;
    default:
      return new MultiDimensionalReadAccess<E>()
      {
        @Override
        public E getElement(int... indexes)
        {
          throw new IndexOutOfBoundsException();
        }

        @Override
        public int getNumDimensions()
        {
          return numDimensions;
        }

        @Override
        public int getSize(int dim)
        {
          if (dim < 0  ||  dim >= numDimensions) {
            throw new IllegalArgumentException("No dimension "+dim);
          }
          return 0;
        }

        @NotNull
        @Override
        public int[] getSizes()
        {
          return new int[numDimensions];
        }

        @Override
        public long getNumElements()
        {
          return 0;
        }

        @Override
        public void visitAll(@NotNull Procedure2<? super E, int[]> visitor)
        {
        }

        @Override
        public void visitAll(@NotNull Procedure1<? super E> visitor)
        {
        }

        @NotNull
        @Override
        public Iterable<E> linearized()
        {
          return Types.emptyIterable();
        }
      };
    }
  }
}
